<?php

namespace Elementor;

if (!defined('ABSPATH'))
    exit;

class Slider_Product extends Widget_Base
{
    public function get_name()
    {
        return 'Slider_Product';
    }

    public function get_title()
    {
        return __('اسلایدر محصولات لمکده', 'lamkadeh');
    }

    public function get_icon()
    {
        return 'eicon-single-product';
    }

    public function get_categories()
    {
        return ['lamkadeh'];
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('محتوا', 'lamkadeh'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'carousel_title',
            [
                'label' => __('نام گردونه', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('جــــــدیدترین محصولات', 'lamkadeh'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'carousel_slug',
            [
                'label' => __('نام انگلیسی گردونه', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => 'LATESTS PRODUCTS',
                'label_block' => true,
            ]
        );

        $this->add_control(
            'all_categories_text',
            [
                'label' => __('متن لینک همه دسته بندی‌ها', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('همه محصولات', 'lamkadeh'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'all_categories_link',
            [
                'label' => __('لینک همه دسته بندی‌ها', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::URL,
                'placeholder' => __('https://example.com/shop', 'lamkadeh'),
                'default' => [
                    'url' => '',
                ],
            ]
        );

        $this->add_control(
            'product_categories',
            [
                'label' => __('انتخاب دسته‌بندی‌ها', 'lamkadeh'),
                'type' => Controls_Manager::SELECT2,
                'options' => $this->get_product_categories(),
                'multiple' => true,
                'default' => [],
                'description' => __('حدکثر ۶ مورد دسته‌بندی انتخاب کنید.', 'lamkadeh'),
            ]
        );

        $this->add_control(
            'products_per_tab',
            [
                'label' => __('تعداد نمایش محصولات هر دسته‌بندی', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 8,
                'step' => 1,
                'default' => 6,
            ]
        );

        $this->end_controls_section();
    }

    private function get_product_categories()
    {
        $terms = get_terms([
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
        ]);

        $options = [];
        if (!empty($terms) && !is_wp_error($terms)) {
            foreach ($terms as $term) {
                $options[$term->term_id] = $term->name;
            }
        }

        return $options;
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $categories = $settings['product_categories'];
?>
        <section class="slider-product">
            <div class="slider-product-head container">
                <div class="slider-product-head-name">
                    <span><?php echo $settings['carousel_slug'] ?></span>
                    <div class="slider-product-head-name-title">
                        <span><?php echo $settings['carousel_title'] ?></span>
                        <svg width="64" height="64" viewBox="0 0 64 64" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <g clip-path="url(#clip0_1_2870)">
                                <circle cx="32" cy="32" r="32" transform="matrix(-1 0 0 1 64 0)" fill="#FFD701" />
                                <path fill-rule="evenodd" clip-rule="evenodd"
                                    d="M42.0002 12.1458C41.4458 12.1458 40.9354 12.2835 40.3765 12.5146C39.8369 12.7378 39.2101 13.0667 38.4297 13.4763L36.5344 14.4709C35.5765 14.9736 34.81 15.3758 34.2181 15.7711C33.6049 16.1806 33.1326 16.6118 32.7887 17.1958C32.4459 17.7779 32.2917 18.4083 32.2178 19.1589C32.146 19.8875 32.146 20.7836 32.146 21.9106V22.0892C32.146 23.2162 32.146 24.1123 32.2178 24.8409C32.2917 25.5916 32.4459 26.2219 32.7887 26.804C33.1326 27.3881 33.6049 27.8192 34.2181 28.2287C34.81 28.624 35.5765 29.0263 36.5344 29.5289L38.4296 30.5235C39.2101 30.9331 39.8369 31.262 40.3765 31.4852C40.9354 31.7163 41.4458 31.8541 42.0002 31.8541C42.5545 31.8541 43.0649 31.7163 43.6238 31.4852C44.1634 31.262 44.7902 30.9331 45.5706 30.5235L47.4659 29.529C48.4238 29.0263 49.1903 28.624 49.7822 28.2287C50.3954 27.8192 50.8677 27.3881 51.2116 26.804C51.5544 26.2219 51.7086 25.5916 51.7826 24.8409C51.8543 24.1123 51.8543 23.2162 51.8543 22.0892V21.9106C51.8543 20.7836 51.8543 19.8875 51.7826 19.1589C51.7086 18.4083 51.5544 17.7779 51.2116 17.1958C50.8677 16.6118 50.3954 16.1806 49.7822 15.7711C49.1903 15.3758 48.4238 14.9735 47.4659 14.4709L45.5707 13.4763C44.7902 13.0667 44.1634 12.7378 43.6238 12.5146C43.0649 12.2835 42.5545 12.1458 42.0002 12.1458ZM39.0401 14.7088C39.8556 14.2808 40.4272 13.9816 40.902 13.7853C41.3642 13.5941 41.6889 13.5208 42.0002 13.5208C42.3114 13.5208 42.6361 13.5941 43.0983 13.7853C43.5732 13.9816 44.1447 14.2808 44.9603 14.7088L46.7936 15.6709C47.7925 16.1951 48.4943 16.5644 49.0186 16.9145C49.2773 17.0873 49.4806 17.2477 49.6447 17.409L46.5915 18.9356L38.8 14.8348L39.0401 14.7088ZM37.3661 15.5873L37.2067 15.6709C36.2078 16.1951 35.5061 16.5644 34.9817 16.9145C34.723 17.0873 34.5198 17.2477 34.3557 17.409L42.0002 21.2313L45.0776 19.6926L37.555 15.7333C37.4819 15.6948 37.4186 15.6451 37.3661 15.5873ZM33.693 18.615C33.6467 18.8111 33.6118 19.0337 33.5861 19.2937C33.5218 19.9471 33.521 20.7756 33.521 21.9463V22.0535C33.521 23.2242 33.5218 24.0527 33.5861 24.7061C33.6491 25.3452 33.7687 25.7585 33.9735 26.1063C34.1772 26.4522 34.4718 26.7448 34.9817 27.0853C35.5061 27.4355 36.2078 27.8047 37.2067 28.3289L39.0401 29.291C39.8556 29.719 40.4272 30.0182 40.902 30.2146C41.052 30.2766 41.1874 30.3262 41.3127 30.3649V22.4248L33.693 18.615ZM42.6877 30.3649C42.8129 30.3262 42.9484 30.2766 43.0983 30.2146C43.5732 30.0182 44.1447 29.719 44.9603 29.291L46.7936 28.3289C47.7925 27.8047 48.4943 27.4355 49.0186 27.0853C49.5285 26.7448 49.8231 26.4522 50.0268 26.1063C50.2316 25.7585 50.3512 25.3452 50.4142 24.7061C50.4786 24.0527 50.4793 23.2242 50.4793 22.0535V21.9463C50.4793 20.7756 50.4786 19.9471 50.4142 19.2937C50.3886 19.0337 50.3536 18.8111 50.3073 18.615L47.271 20.1331V22.9166C47.271 23.2963 46.9632 23.6041 46.5835 23.6041C46.2038 23.6041 45.896 23.2963 45.896 22.9166V20.8206L42.6877 22.4248V30.3649Z"
                                    fill="#1D2977" />
                                <circle cx="22" cy="60" r="17" fill="#F7F8FD" fill-opacity="0.64" />
                            </g>
                            <defs>
                                <clipPath id="clip0_1_2870">
                                    <rect width="64" height="64" rx="32" fill="white" />
                                </clipPath>
                            </defs>
                        </svg>
                    </div>
                    <div class="slider-product-head-shape"></div>
                </div>
                <div class="slider-product-head-line"></div>
                <?php
                if (empty($categories)) {
                    echo '<p>' . __('لطفاً دسته‌بندی محصولات را انتخاب کنید.', 'lamkadeh') . '</p>';
                    return;
                }
                ?>
                <div class="slider-product-head-tabs">
                    <?php
                    foreach ($categories as $index => $cat_id) {
                        $term = get_term($cat_id, 'product_cat');
                        if (!$term || is_wp_error($term))
                            continue;
                    ?>
                        <div class="swiper-slide">
                            <div class="slider-product-head-tab-item <?php echo ($index === 0) ? 'active' : ''; ?>"
                                data-tab-product="tab-product-<?php echo esc_attr($term->term_id) ?>">
                                <span><?php echo esc_html($term->name) ?></span>
                            </div>
                        </div>
                    <?php
                    }
                    ?>
                </div>
                <div class="slider-product-head-line"></div>
                <a href="<?php echo esc_url($settings['all_categories_link']['url']) ?>" class="slider-product-head-link">
                    <div><?php echo $settings['all_categories_text'] ?></div>
                    <svg width="56" height="56" viewBox="0 0 56 56" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <rect width="56" height="56" rx="28" fill="#1D2977" />
                        <path
                            d="M31.6216 21.6066C31.4648 21.7633 31.2504 21.8623 31.0029 21.8623L21.8623 21.8623L21.8623 31.0029C21.8623 31.4813 21.4664 31.8773 20.9879 31.8773C20.5094 31.8773 20.1134 31.4813 20.1134 31.0029L20.1134 20.9879C20.1134 20.5094 20.5094 20.1134 20.9879 20.1134L31.0029 20.1134C31.4813 20.1134 31.8773 20.5094 31.8773 20.9879C31.8856 21.2271 31.7783 21.4498 31.6216 21.6066Z"
                            fill="white" />
                        <path
                            d="M35.6308 35.6309C35.2926 35.9692 34.7316 35.9692 34.3934 35.6309L20.5093 21.7469C20.1711 21.4087 20.1711 20.8477 20.5093 20.5095C20.8476 20.1712 21.4085 20.1712 21.7468 20.5095L35.6308 34.3935C35.9691 34.7317 35.9691 35.2927 35.6308 35.6309Z"
                            fill="white" />
                    </svg>
                </a>
            </div>
            <?php
            foreach ($categories as $index => $cat_id) {
                $term = get_term($cat_id, 'product_cat');
                if (!$term || is_wp_error($term))
                    continue;
                $args = [
                    'post_type' => 'product',
                    'posts_per_page' => (int) $settings['products_per_tab'],
                    'tax_query' => [
                        [
                            'taxonomy' => 'product_cat',
                            'field' => 'term_id',
                            'terms' => (int) $cat_id,
                        ],
                    ],
                    'no_found_rows'          => true,
                    'update_post_meta_cache' => false,
                    'update_post_term_cache' => false,
                ];
                $query = new \WP_Query($args);
            ?>
                <div class="slider-product-main container" id="tab-product-<?php echo esc_attr($term->term_id) ?>">
                    <div class="slider-product-main-rigth">
                        <div class="slider-product-main-card">
                            <div id="prev-slider" class="slider-product-prev">
                                <svg width="44" height="48" viewBox="0 0 44 48" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <circle cx="22" cy="22" r="20.5" transform="matrix(4.37114e-08 -1 -1 -4.37114e-08 44 44)"
                                        fill="#1D2977" stroke="white" stroke-width="3" />
                                    <path
                                        d="M16.1125 28.7206C16.3025 28.7206 16.4925 28.6506 16.6425 28.5006L22.1825 22.9606L27.7225 28.5006C28.0125 28.7906 28.4925 28.7906 28.7825 28.5006C29.0725 28.2106 29.0725 27.7306 28.7825 27.4406L22.7125 21.3706C22.4225 21.0806 21.9425 21.0806 21.6525 21.3706L15.5825 27.4406C15.2925 27.7306 15.2925 28.2106 15.5825 28.5006C15.7225 28.6506 15.9225 28.7206 16.1125 28.7206Z"
                                        fill="#F7F8FD" />
                                    <path
                                        d="M22.1826 47.9999C22.5926 47.9999 22.9326 47.6599 22.9326 47.2499L22.9326 22.4199C22.9326 22.0099 22.5926 21.6699 22.1826 21.6699C21.7726 21.6699 21.4326 22.0099 21.4326 22.4199L21.4326 47.2499C21.4326 47.6599 21.7726 47.9999 22.1826 47.9999Z"
                                        fill="#F7F8FD" />
                                </svg>
                            </div>
                            <?php
                            if ($query->have_posts()):
                            ?>
                                <div class="slider-product-content-cards">
                                    <?php while ($query->have_posts()):
                                        $query->the_post();
                                        global $product;
                                        if ($product->is_type('variable')) {
                                            $regular_price = $product->get_variation_regular_price('max');
                                            $sale_price = $product->get_variation_sale_price('max');
                                        } else {
                                            $regular_price = $product->get_regular_price();
                                            $sale_price = $product->get_sale_price();
                                        }

                                        $product_id = (int) $product->get_id();
                                        $remaining = 0;
                                        $total_sold = 0;
                                        $percentage_sold = 0;
                                        $special_stock = 0;

                                        if ($product->is_type('simple')) {
                                            // تحلیل برای محصول ساده
                                            $special_stock = (int) get_post_meta($product_id, '_lamkadeh_special_stock', true);

                                            if ($special_stock >= 1) {
                                                $total_sold = (int) $product->get_total_sales();
                                                // مدیریت حالت که فروش بیشتر از موجودی نباشد
                                                $total_sold = min($total_sold, $special_stock);
                                                $remaining = max(0, $special_stock - $total_sold);

                                                // فقط اگر موجودی باقی مانده باشد درصد محاسبه شود
                                                if ($remaining > 0) {
                                                    $percentage_sold = ($special_stock > 0) ? ($total_sold / $special_stock) * 100 : 0;
                                                    $percentage_sold = min(100, max(0, round($percentage_sold, 2)));
                                                }
                                            }
                                        } elseif ($product->is_type('variable')) {
                                            // تحلیل برای محصول متغیر - حالت MAX
                                            $variation_ids = $product->get_children();
                                            $max_remaining = 0;

                                            foreach ($variation_ids as $variation_id) {
                                                $variation_special_stock = (int) get_post_meta($variation_id, '_lamkadeh_special_stock', true);

                                                if ($variation_special_stock >= 1) {
                                                    $variation_total_sold = (int) get_post_meta($variation_id, 'total_sales', true);
                                                    $variation_total_sold = min($variation_total_sold, $variation_special_stock);
                                                    $variation_remaining = max(0, $variation_special_stock - $variation_total_sold);

                                                    // پیدا کردن واریاسیونی که بیشترین باقیمانده را دارد
                                                    if ($variation_remaining > $max_remaining) {
                                                        $max_remaining = $variation_remaining;
                                                        $special_stock = $variation_special_stock;
                                                        $total_sold = $variation_total_sold;
                                                        $remaining = $variation_remaining;
                                                    }
                                                }
                                            }

                                            // محاسبه درصد برای حالت MAX
                                            if ($remaining > 0 && $special_stock > 0) {
                                                $percentage_sold = ($total_sold / $special_stock) * 100;
                                                $percentage_sold = min(100, max(0, round($percentage_sold, 2)));
                                            }
                                        }

                                        $discount_percent = '';
                                        if ($regular_price > 0 && $sale_price > 0) {
                                            $discount_percent = round((($regular_price - $sale_price) / $regular_price) * 100);
                                        }

                                        $average = $product->get_average_rating();
                                        $rating_count = $product->get_rating_count();

                                        $gallery_ids = $product->get_gallery_image_ids();
                                        $gallery_urls = [];
                                        $i = 0;
                                        foreach ($gallery_ids as $img_id) {
                                            if ($i >= 5)
                                                break;
                                            $gallery_urls[] = wp_get_attachment_image_url($img_id, 'small');
                                            $i++;
                                        }
                                        if (empty($gallery_urls)) {
                                            $gallery_urls[] = wp_get_attachment_image_url($product->get_image_id(), 'small');
                                        }
                                        $user_id = get_current_user_id();
                                        $wishlist = get_user_meta($user_id, 'lamkadeh_wishlist', true);
                                        $is_favorite = is_array($wishlist) && in_array($product->get_id(), $wishlist);
                                    ?>
                                        <div class="slider-product-card active" data-id="<?php echo esc_attr($product->get_id()); ?>"
                                            data-link="<?php echo esc_url(get_the_permalink()); ?>"
                                            data-title="<?php echo esc_attr(get_the_title()); ?>"
                                            data-price="<?php echo esc_attr($regular_price); ?>"
                                            data-sale-price="<?php echo esc_attr($sale_price); ?>"
                                            data-discount="<?php echo esc_attr($discount_percent); ?>"
                                            data-rating="<?php echo esc_attr($average); ?>"
                                            data-reviews="<?php echo esc_attr($rating_count); ?>"
                                            data-badge="<?php echo __("جدید", "lamkadeh") ?>"
                                            data-sold="<?php echo esc_attr($total_sold); ?>"
                                            data-remaining="<?php echo esc_attr($remaining); ?>"
                                            data-percentage="<?php echo esc_attr($percentage_sold); ?>"
                                            data-gallery='<?php echo json_encode($gallery_urls, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE); ?>'
                                            data-is-favorite="<?php echo $is_favorite ? '1' : ''; ?>">
                                            <div class="slider-product-card-image">
                                                <div class="slider-product-card-image-shape"></div>
                                                <?php echo woocommerce_get_product_thumbnail(); ?>
                                            </div>
                                            <div class="slider-product-card-content">
                                                <div class="slider-product-card-content-name">
                                                    <a href="<?php echo esc_url(get_the_permalink()); ?>">
                                                        <h4><?php echo get_the_title() ?></h4>
                                                    </a>
                                                    <div class="star-review">
                                                        <span><?php echo esc_attr($average); ?></span>
                                                        <svg viewBox="0 0 12 11" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path
                                                                d="M5.25835 0.437178C5.41984 -0.0598314 6.12297 -0.0598298 6.28446 0.43718L7.26517 3.4555C7.33739 3.67776 7.54452 3.82825 7.77823 3.82825H10.9519C11.4745 3.82825 11.6917 4.49697 11.269 4.80414L8.70143 6.66957C8.51235 6.80694 8.43324 7.05043 8.50546 7.2727L9.48617 10.291C9.64766 10.788 9.07881 11.2013 8.65602 10.8941L6.08849 9.02873C5.89942 8.89136 5.64339 8.89136 5.45432 9.02873L2.88679 10.8941C2.464 11.2013 1.89516 10.788 2.05665 10.291L3.03735 7.2727C3.10957 7.05043 3.03046 6.80693 2.84138 6.66956L0.273852 4.80414C-0.14893 4.49697 0.068352 3.82825 0.590939 3.82825H3.76458C3.99829 3.82825 4.20542 3.67776 4.27764 3.45549L5.25835 0.437178Z"
                                                                fill="#FAA307" />
                                                        </svg>
                                                    </div>
                                                </div>
                                                <div class="underline-lamkadeh">
                                                    <div class="underline-lamkadeh-line"></div>
                                                    <div class="underline-lamkadeh-circle"></div>
                                                </div>
                                                <div class="slider-product-card-content-price">
                                                    <div class="slider-product-card-amount">
                                                        <div class="slider-product-card-amount-salesoff">
                                                            <div class="slider-product-card-amount-salesoff-badge">
                                                                <span>%<?php echo esc_html($discount_percent); ?></span>
                                                            </div>
                                                            <span><?php echo number_format((float) $regular_price); ?></span>
                                                            <span class="slider-product-card-amount-salesoff-linedel"></span>
                                                        </div>
                                                        <div class="slider-product-card-amount-regular">
                                                            <span><?php echo number_format((float) $sale_price); ?></span>
                                                            <span>تــــو مان</span>
                                                        </div>
                                                    </div>
                                                    <div class="slider-product-card-shipping">
                                                        <span>ارسال فوری</span>
                                                        <svg width="18" height="18" viewBox="0 0 18 18" fill="none"
                                                            xmlns="http://www.w3.org/2000/svg">
                                                            <g clip-path="url(#clip0_1_2990)">
                                                                <path fill-rule="evenodd" clip-rule="evenodd"
                                                                    d="M8.95769 0.9375H9.04231C10.4206 0.937488 11.5124 0.937479 12.3668 1.05235C13.2461 1.17057 13.9578 1.41966 14.5191 1.98093C15.0803 2.54221 15.3294 3.25392 15.4476 4.13323C15.5247 4.70628 15.5501 5.38608 15.5584 6.1875H15.75C16.4749 6.1875 17.0625 6.77513 17.0625 7.5V8.25C17.0625 8.66312 16.868 9.05213 16.5375 9.3L15.5607 10.0326C15.5552 10.957 15.5335 11.7282 15.4476 12.3668C15.3294 13.2461 15.0803 13.9578 14.5191 14.5191C14.3767 14.6615 14.2246 14.7838 14.0625 14.889V15.75C14.0625 16.4749 13.4749 17.0625 12.75 17.0625H11.625C10.9001 17.0625 10.3125 16.4749 10.3125 15.75V15.5584C9.91901 15.5625 9.4962 15.5625 9.04231 15.5625H8.95769C8.5038 15.5625 8.08099 15.5625 7.6875 15.5584V15.75C7.6875 16.4749 7.09987 17.0625 6.375 17.0625H5.25C4.52513 17.0625 3.9375 16.4749 3.9375 15.75V14.889C3.77544 14.7838 3.62334 14.6615 3.48093 14.5191C2.91966 13.9578 2.67057 13.2461 2.55235 12.3668C2.46649 11.7282 2.44481 10.957 2.43934 10.0326L1.4625 9.3C1.132 9.05213 0.9375 8.66312 0.9375 8.25V7.5C0.9375 6.77513 1.52513 6.1875 2.25 6.1875H2.44159C2.44994 5.38608 2.47531 4.70628 2.55235 4.13323C2.67057 3.25392 2.91966 2.54221 3.48093 1.98093C4.0422 1.41966 4.75392 1.17057 5.63323 1.05235C6.48764 0.937479 7.57937 0.937488 8.95769 0.9375ZM2.4375 7.3125H2.25C2.14645 7.3125 2.0625 7.39645 2.0625 7.5V8.25C2.0625 8.30902 2.09029 8.36459 2.1375 8.4L2.4375 8.625L2.4375 7.45769C2.4375 7.40894 2.4375 7.36054 2.4375 7.3125ZM3.5668 10.3125C3.57496 11.0877 3.59865 11.7061 3.66732 12.2169C3.76877 12.9714 3.95902 13.4062 4.27643 13.7236C4.59383 14.041 5.02857 14.2312 5.78314 14.3327C6.55388 14.4363 7.56988 14.4375 9 14.4375C10.4301 14.4375 11.4461 14.4363 12.2169 14.3327C12.9714 14.2312 13.4062 14.041 13.7236 13.7236C14.041 13.4062 14.2312 12.9714 14.3327 12.2169C14.4013 11.7061 14.425 11.0877 14.4332 10.3125H3.5668ZM14.4375 9.1875H3.56252C3.5625 9.1258 3.5625 9.06331 3.5625 9V7.5C3.5625 6.08797 3.56366 5.07965 3.66344 4.3125H14.3366C14.4363 5.07965 14.4375 6.08797 14.4375 7.5V9C14.4375 9.06331 14.4375 9.1258 14.4375 9.1875ZM15.5625 8.625L15.8625 8.4C15.9097 8.36459 15.9375 8.30902 15.9375 8.25V7.5C15.9375 7.39645 15.8536 7.3125 15.75 7.3125H15.5625C15.5625 7.36054 15.5625 7.40894 15.5625 7.45769V8.625ZM14.0258 3.1875C13.9412 3.02754 13.8411 2.89394 13.7236 2.77643C13.4062 2.45902 12.9714 2.26877 12.2169 2.16732C11.4461 2.0637 10.4301 2.0625 9 2.0625C7.56988 2.0625 6.55388 2.0637 5.78314 2.16732C5.02857 2.26877 4.59383 2.45902 4.27643 2.77643C4.15891 2.89394 4.05883 3.02754 3.97422 3.1875H14.0258ZM5.0625 15.3453V15.75C5.0625 15.8536 5.14645 15.9375 5.25 15.9375H6.375C6.47855 15.9375 6.5625 15.8536 6.5625 15.75V15.5282C6.22549 15.5112 5.91634 15.4857 5.63323 15.4476C5.43449 15.4209 5.24431 15.3875 5.0625 15.3453ZM11.4375 15.5282V15.75C11.4375 15.8536 11.5214 15.9375 11.625 15.9375H12.75C12.8536 15.9375 12.9375 15.8536 12.9375 15.75V15.3453C12.7557 15.3875 12.5655 15.4209 12.3668 15.4476C12.0837 15.4857 11.7745 15.5112 11.4375 15.5282ZM4.6875 12C4.6875 11.6893 4.93934 11.4375 5.25 11.4375H6.375C6.68566 11.4375 6.9375 11.6893 6.9375 12C6.9375 12.3107 6.68566 12.5625 6.375 12.5625H5.25C4.93934 12.5625 4.6875 12.3107 4.6875 12ZM11.0625 12C11.0625 11.6893 11.3143 11.4375 11.625 11.4375H12.75C13.0607 11.4375 13.3125 11.6893 13.3125 12C13.3125 12.3107 13.0607 12.5625 12.75 12.5625H11.625C11.3143 12.5625 11.0625 12.3107 11.0625 12Z"
                                                                    fill="#020308" fill-opacity="0.64" />
                                                            </g>
                                                            <defs>
                                                                <clipPath id="clip0_1_2990">
                                                                    <rect width="18" height="18" fill="white" />
                                                                </clipPath>
                                                            </defs>
                                                        </svg>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php
                                        wp_reset_postdata();
                                    endwhile; ?>
                                </div>
                            <?php
                            endif;
                            ?>
                            <div id="next-slider" class="slider-product-next">
                                <svg width="44" height="48" viewBox="0 0 44 48" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <circle cx="22" cy="26" r="20.5" transform="rotate(90 22 26)" fill="#1D2977" stroke="white"
                                        stroke-width="3" />
                                    <path
                                        d="M16.1125 19.2794C16.3025 19.2794 16.4925 19.3494 16.6425 19.4994L22.1825 25.0394L27.7225 19.4994C28.0125 19.2094 28.4925 19.2094 28.7825 19.4994C29.0725 19.7894 29.0725 20.2694 28.7825 20.5594L22.7125 26.6294C22.4225 26.9194 21.9425 26.9194 21.6525 26.6294L15.5825 20.5594C15.2925 20.2694 15.2925 19.7894 15.5825 19.4994C15.7225 19.3494 15.9225 19.2794 16.1125 19.2794Z"
                                        fill="#F7F8FD" />
                                    <path
                                        d="M22.1826 7.81357e-05C22.5926 7.81716e-05 22.9326 0.340078 22.9326 0.750078L22.9326 25.5801C22.9326 25.9901 22.5926 26.3301 22.1826 26.3301C21.7726 26.3301 21.4326 25.9901 21.4326 25.5801L21.4326 0.750078C21.4326 0.340078 21.7726 7.80999e-05 22.1826 7.81357e-05Z"
                                        fill="#F7F8FD" />
                                </svg>
                            </div>
                        </div>
                    </div>
                    <div class="slider-product-main-left">
                        <div class="slider-product-main-content">
                            <div class="slider-product-main-wrapper">
                                <div class="slider-product-main-wrapper-header">
                                    <div class="slider-product-main-wrapper-header-right">
                                        <div class="slider-product-main-wrapper-circle"></div>
                                        <img class="product-image"
                                            src="<?php echo get_template_directory_uri() ?>/assets/images/products/chair.png"
                                            alt="" />
                                        <svg width="40" height="56" viewBox="0 0 40 56" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M12.9545 25.3675C11.5236 25.3675 10.3636 24.2318 10.3636 22.8308C10.3636 21.4298 11.5236 20.294 12.9545 20.294C14.3855 20.294 15.5455 21.4298 15.5455 22.8308C15.5455 24.2318 14.3855 25.3675 12.9545 25.3675Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M33.6818 25.3675C32.2509 25.3675 31.0909 24.2318 31.0909 22.8308C31.0909 21.4298 32.2509 20.294 33.6818 20.294C35.1127 20.294 36.2727 21.4298 36.2727 22.8308C36.2727 24.2318 35.1127 25.3675 33.6818 25.3675Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M44.0455 25.3675C42.6145 25.3675 41.4545 24.2318 41.4545 22.8308C41.4545 21.4298 42.6145 20.294 44.0455 20.294C45.4764 20.294 46.6364 21.4298 46.6364 22.8308C46.6364 24.2318 45.4764 25.3675 44.0455 25.3675Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M2.59091 15.2205C1.15999 15.2205 5.09439e-07 14.0848 4.47634e-07 12.6838C3.85828e-07 11.2828 1.15999 10.147 2.59091 10.147C4.02183 10.147 5.18182 11.2828 5.18182 12.6838C5.18182 14.0848 4.02183 15.2205 2.59091 15.2205Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M12.9545 15.2205C11.5236 15.2205 10.3636 14.0848 10.3636 12.6838C10.3636 11.2828 11.5236 10.147 12.9545 10.147C14.3855 10.147 15.5455 11.2828 15.5455 12.6838C15.5455 14.0848 14.3855 15.2205 12.9545 15.2205Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M23.3182 15.2205C21.8873 15.2205 20.7273 14.0848 20.7273 12.6838C20.7273 11.2828 21.8873 10.147 23.3182 10.147C24.7491 10.147 25.9091 11.2828 25.9091 12.6838C25.9091 14.0848 24.7491 15.2205 23.3182 15.2205Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M44.0455 15.2205C42.6145 15.2205 41.4545 14.0848 41.4545 12.6838C41.4545 11.2828 42.6145 10.147 44.0455 10.147C45.4764 10.147 46.6364 11.2828 46.6364 12.6838C46.6364 14.0848 45.4764 15.2205 44.0455 15.2205Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M54.4091 15.2205C52.9782 15.2205 51.8182 14.0848 51.8182 12.6838C51.8182 11.2828 52.9782 10.147 54.4091 10.147C55.84 10.147 57 11.2828 57 12.6838C57 14.0848 55.84 15.2205 54.4091 15.2205Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M2.59091 5.07351C1.15999 5.07351 6.18054e-08 3.93776 0 2.53675C-6.18054e-08 1.13574 1.15999 2.3063e-06 2.59091 2.24433e-06C4.02183 2.18235e-06 5.18182 1.13574 5.18182 2.53675C5.18182 3.93776 4.02183 5.0735 2.59091 5.07351Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M23.3182 5.0735C21.8873 5.0735 20.7273 3.93776 20.7273 2.53675C20.7273 1.13574 21.8873 1.40857e-06 23.3182 1.3466e-06C24.7491 1.28462e-06 25.9091 1.13574 25.9091 2.53675C25.9091 3.93776 24.7491 5.0735 23.3182 5.0735Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M33.6818 5.0735C32.2509 5.0735 31.0909 3.93776 31.0909 2.53675C31.0909 1.13574 32.2509 9.59706e-07 33.6818 8.97731e-07C35.1127 8.35756e-07 36.2727 1.13574 36.2727 2.53675C36.2727 3.93776 35.1127 5.0735 33.6818 5.0735Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M44.0455 5.0735C42.6145 5.0735 41.4545 3.93776 41.4545 2.53675C41.4545 1.13574 42.6145 5.1084e-07 44.0455 4.48865e-07C45.4764 3.8689e-07 46.6364 1.13574 46.6364 2.53675C46.6364 3.93776 45.4764 5.0735 44.0455 5.0735Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M54.4091 5.0735C52.9782 5.0735 51.8182 3.93776 51.8182 2.53675C51.8182 1.13574 52.9782 6.19754e-08 54.4091 0C55.84 -6.19754e-08 57 1.13574 57 2.53675C57 3.93776 55.84 5.0735 54.4091 5.0735Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M33.6818 15.2205C32.2509 15.2205 31.0909 14.0848 31.0909 12.6838C31.0909 11.2828 32.2509 10.147 33.6818 10.147C35.1127 10.147 36.2727 11.2828 36.2727 12.6838C36.2727 14.0848 35.1127 15.2205 33.6818 15.2205Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M12.9545 5.0735C11.5236 5.0735 10.3636 3.93776 10.3636 2.53675C10.3636 1.13574 11.5236 1.85744e-06 12.9545 1.79546e-06C14.3855 1.73349e-06 15.5455 1.13574 15.5455 2.53675C15.5455 3.93776 14.3855 5.0735 12.9545 5.0735Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M23.3182 25.3675C21.8873 25.3675 20.7273 24.2318 20.7273 22.8308C20.7273 21.4298 21.8873 20.294 23.3182 20.294C24.7491 20.294 25.9091 21.4298 25.9091 22.8308C25.9091 24.2318 24.7491 25.3675 23.3182 25.3675Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M54.4091 25.3675C52.9782 25.3675 51.8182 24.2318 51.8182 22.8308C51.8182 21.4298 52.9782 20.294 54.4091 20.294C55.84 20.294 57 21.4298 57 22.8308C57 24.2318 55.84 25.3675 54.4091 25.3675Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M2.59091 25.3675C1.15999 25.3675 9.57074e-07 24.2318 8.95268e-07 22.8308C8.33463e-07 21.4298 1.15999 20.294 2.59091 20.294C4.02183 20.294 5.18182 21.4298 5.18182 22.8308C5.18182 24.2318 4.02183 25.3675 2.59091 25.3675Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M12.9545 56C11.5236 56 10.3636 54.8643 10.3636 53.4632C10.3636 52.0622 11.5236 50.9265 12.9545 50.9265C14.3855 50.9265 15.5455 52.0622 15.5455 53.4632C15.5455 54.8643 14.3855 56 12.9545 56Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M33.6818 56C32.2509 56 31.0909 54.8643 31.0909 53.4632C31.0909 52.0622 32.2509 50.9265 33.6818 50.9265C35.1127 50.9265 36.2727 52.0622 36.2727 53.4632C36.2727 54.8643 35.1127 56 33.6818 56Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M44.0455 56C42.6145 56 41.4545 54.8643 41.4545 53.4632C41.4545 52.0622 42.6145 50.9265 44.0455 50.9265C45.4764 50.9265 46.6364 52.0622 46.6364 53.4632C46.6364 54.8643 45.4764 56 44.0455 56Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M2.59091 45.853C1.15999 45.853 5.09439e-07 44.7173 4.47634e-07 43.3162C3.85828e-07 41.9152 1.15999 40.7795 2.59091 40.7795C4.02183 40.7795 5.18182 41.9152 5.18182 43.3162C5.18182 44.7173 4.02183 45.853 2.59091 45.853Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M12.9545 45.853C11.5236 45.853 10.3636 44.7173 10.3636 43.3162C10.3636 41.9152 11.5236 40.7795 12.9545 40.7795C14.3855 40.7795 15.5455 41.9152 15.5455 43.3162C15.5455 44.7173 14.3855 45.853 12.9545 45.853Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M23.3182 45.853C21.8873 45.853 20.7273 44.7173 20.7273 43.3162C20.7273 41.9152 21.8873 40.7795 23.3182 40.7795C24.7491 40.7795 25.9091 41.9152 25.9091 43.3162C25.9091 44.7173 24.7491 45.853 23.3182 45.853Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M44.0455 45.853C42.6145 45.853 41.4545 44.7173 41.4545 43.3162C41.4545 41.9152 42.6145 40.7795 44.0455 40.7795C45.4764 40.7795 46.6364 41.9152 46.6364 43.3162C46.6364 44.7173 45.4764 45.853 44.0455 45.853Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M54.4091 45.853C52.9782 45.853 51.8182 44.7173 51.8182 43.3162C51.8182 41.9152 52.9782 40.7795 54.4091 40.7795C55.84 40.7795 57 41.9152 57 43.3162C57 44.7173 55.84 45.853 54.4091 45.853Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M2.59091 35.706C1.15999 35.706 6.18054e-08 34.5702 0 33.1692C-6.18054e-08 31.7682 1.15999 30.6325 2.59091 30.6325C4.02183 30.6325 5.18182 31.7682 5.18182 33.1692C5.18182 34.5702 4.02183 35.706 2.59091 35.706Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M23.3182 35.706C21.8873 35.706 20.7273 34.5702 20.7273 33.1692C20.7273 31.7682 21.8873 30.6325 23.3182 30.6325C24.7491 30.6325 25.9091 31.7682 25.9091 33.1692C25.9091 34.5702 24.7491 35.706 23.3182 35.706Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M33.6818 35.706C32.2509 35.706 31.0909 34.5702 31.0909 33.1692C31.0909 31.7682 32.2509 30.6325 33.6818 30.6325C35.1127 30.6325 36.2727 31.7682 36.2727 33.1692C36.2727 34.5702 35.1127 35.706 33.6818 35.706Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M44.0455 35.706C42.6145 35.706 41.4545 34.5702 41.4545 33.1692C41.4545 31.7682 42.6145 30.6325 44.0455 30.6325C45.4764 30.6325 46.6364 31.7682 46.6364 33.1692C46.6364 34.5702 45.4764 35.706 44.0455 35.706Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M54.4091 35.706C52.9782 35.706 51.8182 34.5702 51.8182 33.1692C51.8182 31.7682 52.9782 30.6325 54.4091 30.6325C55.84 30.6325 57 31.7682 57 33.1692C57 34.5702 55.84 35.706 54.4091 35.706Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M33.6818 45.853C32.2509 45.853 31.0909 44.7173 31.0909 43.3162C31.0909 41.9152 32.2509 40.7795 33.6818 40.7795C35.1127 40.7795 36.2727 41.9152 36.2727 43.3162C36.2727 44.7173 35.1127 45.853 33.6818 45.853Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M12.9545 35.706C11.5236 35.706 10.3636 34.5702 10.3636 33.1692C10.3636 31.7682 11.5236 30.6325 12.9545 30.6325C14.3855 30.6325 15.5455 31.7682 15.5455 33.1692C15.5455 34.5702 14.3855 35.706 12.9545 35.706Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M23.3182 56C21.8873 56 20.7273 54.8643 20.7273 53.4632C20.7273 52.0622 21.8873 50.9265 23.3182 50.9265C24.7491 50.9265 25.9091 52.0622 25.9091 53.4632C25.9091 54.8643 24.7491 56 23.3182 56Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M54.4091 56C52.9782 56 51.8182 54.8643 51.8182 53.4632C51.8182 52.0622 52.9782 50.9265 54.4091 50.9265C55.84 50.9265 57 52.0622 57 53.4632C57 54.8643 55.84 56 54.4091 56Z" fill="#F7F8FD" fill-opacity="0.4" />
                                            <path d="M2.59091 56C1.15999 56 9.57074e-07 54.8643 8.95268e-07 53.4633C8.33463e-07 52.0622 1.15999 50.9265 2.59091 50.9265C4.02183 50.9265 5.18182 52.0622 5.18182 53.4633C5.18182 54.8643 4.02183 56 2.59091 56Z" fill="#F7F8FD" fill-opacity="0.4" />
                                        </svg>
                                        <div class="slider-product-main-wrapper-circle-submask"></div>
                                        <div class="slider-product-main-wrapper-circle-dashed"></div>
                                    </div>
                                    <div class="slider-product-main-wrapper-header-left">
                                        <div class="slider-product-main-wrapper-header-left-name">
                                            <div class="slider-product-main-wrapper-header-left-review">
                                                <div>
                                                    <span class="product-reviews">(۳۶۳) نظر</span>
                                                    <div class="slider-product-main-wrapper-header-left-review-line"></div>
                                                    <div class="slider-product-main-wrapper-header-left-review-stars"></div>
                                                </div>
                                                <div class="slider-product-main-wrapper-header-left-review-badge">
                                                    <span class="product-discount">جدید</span>
                                                </div>
                                            </div>
                                            <a class="product-link" href="#">
                                                <h4 class="product-title">صندلی دکوری با پارچه ضدلک</h4>
                                            </a>
                                        </div>
                                        <div class="underline-lamkadeh">
                                            <div class="underline-lamkadeh-line"></div>
                                            <div class="underline-lamkadeh-circle"></div>
                                        </div>
                                        <div class="slider-product-main-wrapper-header-left-price">
                                            <div class="slider-product-main-wrapper-header-left-price-regular">
                                                <div class="slider-product-main-wrapper-header-left-price-regular-badge">
                                                    <span class="product-discount-percent">32%</span>
                                                </div>
                                                <span
                                                    class="slider-product-main-wrapper-header-left-price-regular-amount product-regular">۵۶,۰۰۰,۰۰۰</span>
                                                <div class="slider-product-main-wrapper-header-left-price-regular-line"></div>
                                            </div>
                                            <div class="slider-product-main-wrapper-header-left-price-salesoff">
                                                <div class="slider-product-main-wrapper-header-left-price-salesoff-shape"></div>
                                                <span
                                                    class="slider-product-main-wrapper-header-left-price-salesoff-amount product-sale">۱۵۲,۳۰۰,۰۰۰</span>
                                                <span>تــــو مان</span>
                                            </div>
                                        </div>
                                        <div class="slider-product-main-wrapper-header-left-counter">
                                            <div class="slider-product-main-wrapper-header-left-counter-wrapper">
                                                <div class="slider-product-main-wrapper-header-left-counter-progressbar-head">
                                                    <span>فروخته شده:<b>23</b></span>
                                                    <span>باقیمانده: <b>6</b></span>
                                                </div>
                                                <div class="slider-product-main-wrapper-header-left-counter-progressbar-body">
                                                    <div class="slider-product-main-wrapper-header-left-counter-progressbar"
                                                        style="width: 80%;"></div>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="underline-lamkadeh">
                                            <div class="underline-lamkadeh-line"></div>
                                            <div class="underline-lamkadeh-circle"></div>
                                        </div>
                                        <div class="slider-product-main-wrapper-header-left-add-to-cart">
                                            <div class="slider-product-main-wrapper-header-left-add-to-cart-button">
                                                <svg width="112" height="112" viewBox="0 0 56 56" fill="none"
                                                    xmlns="http://www.w3.org/2000/svg">
                                                    <rect y="0.000244141" width="56" height="56" rx="28" fill="#1D2977" />
                                                    <path fill-rule="evenodd" clip-rule="evenodd"
                                                        d="M28.0001 18.7502C26.7574 18.7502 25.7501 19.7576 25.7501 21.0002V21.2547C26.1676 21.2502 26.6183 21.2502 27.1053 21.2502H28.8948C29.3819 21.2502 29.8326 21.2502 30.2501 21.2547V21.0002C30.2501 19.7576 29.2427 18.7502 28.0001 18.7502ZM31.7501 21.3072V21.0002C31.7501 18.9292 30.0711 17.2502 28.0001 17.2502C25.929 17.2502 24.2501 18.9292 24.2501 21.0002V21.3072C24.1151 21.317 23.9848 21.3286 23.859 21.3421C22.9807 21.4368 22.2461 21.6351 21.5938 22.0822C21.3695 22.236 21.1588 22.4087 20.964 22.5986C20.3977 23.1505 20.0591 23.8319 19.794 24.6746C19.5367 25.4928 19.3287 26.5329 19.0673 27.8398L19.0482 27.9352C18.6716 29.8184 18.3748 31.3023 18.2895 32.4842C18.2024 33.6918 18.3242 34.7078 18.8962 35.5882C19.087 35.8819 19.3098 36.1537 19.5604 36.3984C20.3116 37.132 21.2839 37.4506 22.4852 37.6021C23.6609 37.7503 25.1742 37.7503 27.0946 37.7502H28.9055C30.826 37.7503 32.3393 37.7503 33.5149 37.6021C34.7162 37.4506 35.6886 37.132 36.4397 36.3984C36.6903 36.1537 36.9131 35.8819 37.1039 35.5882C37.676 34.7078 37.7977 33.6918 37.7106 32.4842C37.6254 31.3023 37.3286 29.8184 36.9519 27.9353L36.9328 27.8399C36.6715 26.5329 36.4635 25.4928 36.2061 24.6746C35.941 23.8319 35.6025 23.1505 35.0361 22.5986C34.8414 22.4087 34.6306 22.236 34.4063 22.0822C33.754 21.6351 33.0194 21.4368 32.1411 21.3421C32.0153 21.3286 31.8851 21.317 31.7501 21.3072ZM24.0198 22.8335C23.2731 22.914 22.8118 23.066 22.4419 23.3195C22.2884 23.4247 22.1442 23.5429 22.0109 23.6728C21.6898 23.9858 21.4503 24.4083 21.2249 25.1247C20.9946 25.8569 20.8015 26.8175 20.5297 28.1765C20.1401 30.1242 19.8633 31.5155 19.7857 32.5921C19.7091 33.654 19.8384 34.2852 20.154 34.7709C20.2846 34.9719 20.437 35.1579 20.6085 35.3253C21.0229 35.7301 21.6165 35.9807 22.6728 36.1138C23.7437 36.2488 25.1623 36.2502 27.1486 36.2502H28.8515C30.8378 36.2502 32.2564 36.2488 33.3273 36.1138C34.3837 35.9807 34.9772 35.7301 35.3916 35.3253C35.5631 35.1579 35.7155 34.9719 35.8461 34.7709C36.1617 34.2852 36.2911 33.654 36.2145 32.5921C36.1368 31.5155 35.86 30.1242 35.4705 28.1765C35.1987 26.8175 35.0055 25.8569 34.7752 25.1247C34.5498 24.4083 34.3103 23.9858 33.9892 23.6728C33.8559 23.5429 33.7118 23.4247 33.5582 23.3195C33.1884 23.066 32.7271 22.914 31.9803 22.8335C31.2173 22.7513 30.2374 22.7502 28.8515 22.7502H27.1486C25.7627 22.7502 24.7828 22.7513 24.0198 22.8335ZM24.921 30.2931C25.3116 30.1551 25.7401 30.3598 25.8781 30.7503C26.1873 31.6252 27.0218 32.2502 28.0003 32.2502C28.9787 32.2502 29.8132 31.6252 30.1224 30.7503C30.2605 30.3598 30.6889 30.1551 31.0795 30.2931C31.47 30.4311 31.6747 30.8596 31.5367 31.2502C31.0222 32.7057 29.6342 33.7502 28.0003 33.7502C26.3663 33.7502 24.9783 32.7057 24.4638 31.2502C24.3258 30.8596 24.5305 30.4311 24.921 30.2931Z"
                                                        fill="white" />
                                                </svg>
                                                <a href="#">
                                                    افزودن به سبد خرید
                                                </a>
                                            </div>
                                            <div class="slider-product-main-wrapper-header-left-add-to-cart-share">
                                                <div class="product-share-btn" data-id="">
                                                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none"
                                                        xmlns="http://www.w3.org/2000/svg">
                                                        <path fill-rule="evenodd" clip-rule="evenodd"
                                                            d="M16.5 2.25C14.7051 2.25 13.25 3.70507 13.25 5.5C13.25 5.69591 13.2673 5.88776 13.3006 6.07412L8.56991 9.38558C8.54587 9.4024 8.52312 9.42038 8.50168 9.43939C7.94993 9.00747 7.25503 8.75 6.5 8.75C4.70507 8.75 3.25 10.2051 3.25 12C3.25 13.7949 4.70507 15.25 6.5 15.25C7.25503 15.25 7.94993 14.9925 8.50168 14.5606C8.52312 14.5796 8.54587 14.5976 8.56991 14.6144L13.3006 17.9259C13.2673 18.1122 13.25 18.3041 13.25 18.5C13.25 20.2949 14.7051 21.75 16.5 21.75C18.2949 21.75 19.75 20.2949 19.75 18.5C19.75 16.7051 18.2949 15.25 16.5 15.25C15.4472 15.25 14.5113 15.7506 13.9174 16.5267L9.43806 13.3911C9.63809 12.9694 9.75 12.4978 9.75 12C9.75 11.5022 9.63809 11.0306 9.43806 10.6089L13.9174 7.4733C14.5113 8.24942 15.4472 8.75 16.5 8.75C18.2949 8.75 19.75 7.29493 19.75 5.5C19.75 3.70507 18.2949 2.25 16.5 2.25ZM14.75 5.5C14.75 4.5335 15.5335 3.75 16.5 3.75C17.4665 3.75 18.25 4.5335 18.25 5.5C18.25 6.4665 17.4665 7.25 16.5 7.25C15.5335 7.25 14.75 6.4665 14.75 5.5ZM6.5 10.25C5.5335 10.25 4.75 11.0335 4.75 12C4.75 12.9665 5.5335 13.75 6.5 13.75C7.4665 13.75 8.25 12.9665 8.25 12C8.25 11.0335 7.4665 10.25 6.5 10.25ZM16.5 16.75C15.5335 16.75 14.75 17.5335 14.75 18.5C14.75 19.4665 15.5335 20.25 16.5 20.25C17.4665 20.25 18.25 19.4665 18.25 18.5C18.25 17.5335 17.4665 16.75 16.5 16.75Z"
                                                            fill="#1D2977" />
                                                    </svg>
                                                </div>
                                                <?php
                                                if (is_user_logged_in()):
                                                    $user_id = get_current_user_id();
                                                    $wishlist = get_user_meta($user_id, 'lamkadeh_wishlist', true);
                                                    $is_favorite = is_array($wishlist) && in_array($product->get_id(), $wishlist);
                                                ?>
                                                    <div class="wishlist-btn <?php echo $is_favorite ? '1' : ''; ?>" data-id="">
                                                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none"
                                                            xmlns="http://www.w3.org/2000/svg">
                                                            <path fill-rule="evenodd" clip-rule="evenodd"
                                                                d="M5.62436 4.42435C3.96537 5.18268 2.75 6.98639 2.75 9.13725C2.75 11.3347 3.64922 13.0284 4.93829 14.4799C6.00072 15.6763 7.28684 16.6678 8.54113 17.6348C8.83904 17.8644 9.13515 18.0927 9.42605 18.322C9.95208 18.7367 10.4213 19.1007 10.8736 19.365C11.3261 19.6294 11.6904 19.7502 12 19.7502C12.3096 19.7502 12.6739 19.6294 13.1264 19.365C13.5787 19.1007 14.0479 18.7367 14.574 18.322C14.8649 18.0927 15.161 17.8644 15.4589 17.6348C16.7132 16.6678 17.9993 15.6763 19.0617 14.4799C20.3508 13.0284 21.25 11.3347 21.25 9.13725C21.25 6.98639 20.0346 5.18268 18.3756 4.42435C16.7639 3.68763 14.5983 3.88273 12.5404 6.02089C12.399 6.16778 12.2039 6.25079 12 6.25079C11.7961 6.25079 11.601 6.16778 11.4596 6.02089C9.40166 3.88273 7.23607 3.68763 5.62436 4.42435ZM12 4.45897C9.68795 2.3904 7.09896 2.10102 5.00076 3.06012C2.78471 4.07308 1.25 6.42519 1.25 9.13725C1.25 11.8028 2.3605 13.8362 3.81672 15.476C4.98287 16.7891 6.41022 17.8881 7.67083 18.8587C7.95659 19.0788 8.23378 19.2922 8.49742 19.5C9.00965 19.9038 9.55954 20.3344 10.1168 20.6601C10.6739 20.9856 11.3096 21.2502 12 21.2502C12.6904 21.2502 13.3261 20.9856 13.8832 20.6601C14.4405 20.3344 14.9903 19.9038 15.5026 19.5C15.7662 19.2922 16.0434 19.0788 16.3292 18.8587C17.5898 17.8881 19.0171 16.7891 20.1833 15.476C21.6395 13.8362 22.75 11.8028 22.75 9.13725C22.75 6.42519 21.2153 4.07308 18.9992 3.06012C16.901 2.10102 14.3121 2.3904 12 4.45897Z"
                                                                fill="#1D2977" />
                                                        </svg>
                                                    </div>
                                                <?php else: ?>
                                                    <div onclick="loginBeforeWhishlist()">
                                                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none"
                                                            xmlns="http://www.w3.org/2000/svg">
                                                            <path fill-rule="evenodd" clip-rule="evenodd"
                                                                d="M5.62436 4.42435C3.96537 5.18268 2.75 6.98639 2.75 9.13725C2.75 11.3347 3.64922 13.0284 4.93829 14.4799C6.00072 15.6763 7.28684 16.6678 8.54113 17.6348C8.83904 17.8644 9.13515 18.0927 9.42605 18.322C9.95208 18.7367 10.4213 19.1007 10.8736 19.365C11.3261 19.6294 11.6904 19.7502 12 19.7502C12.3096 19.7502 12.6739 19.6294 13.1264 19.365C13.5787 19.1007 14.0479 18.7367 14.574 18.322C14.8649 18.0927 15.161 17.8644 15.4589 17.6348C16.7132 16.6678 17.9993 15.6763 19.0617 14.4799C20.3508 13.0284 21.25 11.3347 21.25 9.13725C21.25 6.98639 20.0346 5.18268 18.3756 4.42435C16.7639 3.68763 14.5983 3.88273 12.5404 6.02089C12.399 6.16778 12.2039 6.25079 12 6.25079C11.7961 6.25079 11.601 6.16778 11.4596 6.02089C9.40166 3.88273 7.23607 3.68763 5.62436 4.42435ZM12 4.45897C9.68795 2.3904 7.09896 2.10102 5.00076 3.06012C2.78471 4.07308 1.25 6.42519 1.25 9.13725C1.25 11.8028 2.3605 13.8362 3.81672 15.476C4.98287 16.7891 6.41022 17.8881 7.67083 18.8587C7.95659 19.0788 8.23378 19.2922 8.49742 19.5C9.00965 19.9038 9.55954 20.3344 10.1168 20.6601C10.6739 20.9856 11.3096 21.2502 12 21.2502C12.6904 21.2502 13.3261 20.9856 13.8832 20.6601C14.4405 20.3344 14.9903 19.9038 15.5026 19.5C15.7662 19.2922 16.0434 19.0788 16.3292 18.8587C17.5898 17.8881 19.0171 16.7891 20.1833 15.476C21.6395 13.8362 22.75 11.8028 22.75 9.13725C22.75 6.42519 21.2153 4.07308 18.9992 3.06012C16.901 2.10102 14.3121 2.3904 12 4.45897Z"
                                                                fill="#1D2977" />
                                                        </svg>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="slider-product-main-wrapper-body">

                                </div>
                            </div>

                            <svg class="slider-product-main-content-svg-one" width="76" height="74" viewBox="0 0 76 74" fill="none"
                                xmlns="http://www.w3.org/2000/svg">
                                <path
                                    d="M17.2727 33.5214C15.3648 33.5214 13.8182 32.0206 13.8182 30.1692C13.8182 28.3179 15.3648 26.8171 17.2727 26.8171C19.1806 26.8171 20.7273 28.3179 20.7273 30.1692C20.7273 32.0206 19.1806 33.5214 17.2727 33.5214Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M44.9091 33.5214C43.0012 33.5214 41.4545 32.0206 41.4545 30.1692C41.4545 28.3179 43.0012 26.8171 44.9091 26.8171C46.817 26.8171 48.3636 28.3179 48.3636 30.1692C48.3636 32.0206 46.817 33.5214 44.9091 33.5214Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.7273 33.5214C56.8194 33.5214 55.2727 32.0206 55.2727 30.1692C55.2727 28.3179 56.8194 26.8171 58.7273 26.8171C60.6352 26.8171 62.1818 28.3179 62.1818 30.1692C62.1818 32.0206 60.6352 33.5214 58.7273 33.5214Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M3.45455 20.1128C1.54665 20.1128 6.79252e-07 18.612 5.96845e-07 16.7607C5.14438e-07 14.9094 1.54665 13.4085 3.45455 13.4085C5.36244 13.4085 6.90909 14.9094 6.90909 16.7607C6.90909 18.612 5.36244 20.1128 3.45455 20.1128Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M17.2727 20.1128C15.3648 20.1128 13.8182 18.612 13.8182 16.7607C13.8182 14.9094 15.3648 13.4085 17.2727 13.4085C19.1806 13.4085 20.7273 14.9094 20.7273 16.7607C20.7273 18.612 19.1806 20.1128 17.2727 20.1128Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M31.0909 20.1128C29.183 20.1128 27.6364 18.612 27.6364 16.7607C27.6364 14.9094 29.183 13.4085 31.0909 13.4085C32.9988 13.4085 34.5455 14.9094 34.5455 16.7607C34.5455 18.612 32.9988 20.1128 31.0909 20.1128Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.7273 20.1128C56.8194 20.1128 55.2727 18.612 55.2727 16.7607C55.2727 14.9094 56.8194 13.4085 58.7273 13.4085C60.6352 13.4085 62.1818 14.9093 62.1818 16.7607C62.1818 18.612 60.6352 20.1128 58.7273 20.1128Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M72.5455 20.1128C70.6376 20.1128 69.0909 18.612 69.0909 16.7607C69.0909 14.9093 70.6376 13.4085 72.5455 13.4085C74.4533 13.4085 76 14.9093 76 16.7607C76 18.612 74.4533 20.1128 72.5455 20.1128Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M3.45455 6.70427C1.54665 6.70427 8.24072e-08 5.20347 0 3.35214C-8.24072e-08 1.5008 1.54665 3.04761e-06 3.45454 2.96572e-06C5.36244 2.88382e-06 6.90909 1.5008 6.90909 3.35214C6.90909 5.20347 5.36244 6.70427 3.45455 6.70427Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M31.0909 6.70427C29.183 6.70427 27.6364 5.20347 27.6364 3.35214C27.6364 1.5008 29.183 1.86133e-06 31.0909 1.77943e-06C32.9988 1.69753e-06 34.5454 1.5008 34.5454 3.35214C34.5454 5.20347 32.9988 6.70427 31.0909 6.70427Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M44.9091 6.70427C43.0012 6.70427 41.4545 5.20347 41.4545 3.35214C41.4545 1.5008 43.0012 1.26818e-06 44.9091 1.18629e-06C46.817 1.10439e-06 48.3636 1.5008 48.3636 3.35214C48.3636 5.20347 46.817 6.70427 44.9091 6.70427Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.7273 6.70427C56.8194 6.70427 55.2727 5.20347 55.2727 3.35214C55.2727 1.5008 56.8194 6.75039e-07 58.7273 5.93143e-07C60.6352 5.11247e-07 62.1818 1.5008 62.1818 3.35214C62.1818 5.20347 60.6352 6.70427 58.7273 6.70427Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M72.5455 6.70427C70.6376 6.70427 69.0909 5.20347 69.0909 3.35214C69.0909 1.5008 70.6376 8.1896e-08 72.5455 0C74.4533 -8.1896e-08 76 1.5008 76 3.35214C76 5.20347 74.4533 6.70427 72.5455 6.70427Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M44.9091 20.1128C43.0012 20.1128 41.4545 18.612 41.4545 16.7607C41.4545 14.9094 43.0012 13.4085 44.9091 13.4085C46.817 13.4085 48.3636 14.9094 48.3636 16.7607C48.3636 18.612 46.817 20.1128 44.9091 20.1128Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M17.2727 6.70427C15.3648 6.70427 13.8182 5.20347 13.8182 3.35214C13.8182 1.5008 15.3648 2.45447e-06 17.2727 2.37257e-06C19.1806 2.29068e-06 20.7273 1.5008 20.7273 3.35214C20.7273 5.20347 19.1806 6.70427 17.2727 6.70427Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M31.0909 33.5214C29.183 33.5214 27.6364 32.0206 27.6364 30.1692C27.6364 28.3179 29.183 26.8171 31.0909 26.8171C32.9988 26.8171 34.5455 28.3179 34.5455 30.1692C34.5455 32.0206 32.9988 33.5214 31.0909 33.5214Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M72.5455 33.5214C70.6376 33.5214 69.0909 32.0206 69.0909 30.1692C69.0909 28.3179 70.6376 26.8171 72.5455 26.8171C74.4533 26.8171 76 28.3179 76 30.1692C76 32.0206 74.4533 33.5214 72.5455 33.5214Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M3.45455 33.5214C1.54665 33.5214 1.2761e-06 32.0206 1.19369e-06 30.1692C1.11128e-06 28.3179 1.54665 26.8171 3.45455 26.8171C5.36244 26.8171 6.90909 28.3179 6.90909 30.1692C6.90909 32.0206 5.36244 33.5214 3.45455 33.5214Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M17.2727 74C15.3648 74 13.8182 72.4992 13.8182 70.6479C13.8182 68.7965 15.3648 67.2957 17.2727 67.2957C19.1806 67.2957 20.7273 68.7965 20.7273 70.6479C20.7273 72.4992 19.1806 74 17.2727 74Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M44.9091 74C43.0012 74 41.4545 72.4992 41.4545 70.6479C41.4545 68.7965 43.0012 67.2957 44.9091 67.2957C46.817 67.2957 48.3636 68.7965 48.3636 70.6479C48.3636 72.4992 46.817 74 44.9091 74Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.7273 74C56.8194 74 55.2727 72.4992 55.2727 70.6479C55.2727 68.7965 56.8194 67.2957 58.7273 67.2957C60.6352 67.2957 62.1818 68.7965 62.1818 70.6479C62.1818 72.4992 60.6352 74 58.7273 74Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M3.45455 60.5915C1.54665 60.5915 6.79252e-07 59.0907 5.96845e-07 57.2393C5.14438e-07 55.388 1.54665 53.8872 3.45455 53.8872C5.36244 53.8872 6.90909 55.388 6.90909 57.2393C6.90909 59.0907 5.36244 60.5915 3.45455 60.5915Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M17.2727 60.5915C15.3648 60.5915 13.8182 59.0907 13.8182 57.2393C13.8182 55.388 15.3648 53.8872 17.2727 53.8872C19.1806 53.8872 20.7273 55.388 20.7273 57.2393C20.7273 59.0907 19.1806 60.5915 17.2727 60.5915Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M31.0909 60.5915C29.183 60.5915 27.6364 59.0907 27.6364 57.2393C27.6364 55.388 29.183 53.8872 31.0909 53.8872C32.9988 53.8872 34.5455 55.388 34.5455 57.2393C34.5455 59.0907 32.9988 60.5915 31.0909 60.5915Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.7273 60.5915C56.8194 60.5915 55.2727 59.0907 55.2727 57.2393C55.2727 55.388 56.8194 53.8872 58.7273 53.8872C60.6352 53.8872 62.1818 55.388 62.1818 57.2393C62.1818 59.0907 60.6352 60.5915 58.7273 60.5915Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M72.5455 60.5915C70.6376 60.5915 69.0909 59.0907 69.0909 57.2393C69.0909 55.388 70.6376 53.8872 72.5455 53.8872C74.4533 53.8872 76 55.388 76 57.2393C76 59.0907 74.4533 60.5915 72.5455 60.5915Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M3.45455 47.1829C1.54665 47.1829 8.24072e-08 45.6821 0 43.8308C-8.24072e-08 41.9794 1.54665 40.4786 3.45454 40.4786C5.36244 40.4786 6.90909 41.9794 6.90909 43.8308C6.90909 45.6821 5.36244 47.1829 3.45455 47.1829Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M31.0909 47.1829C29.183 47.1829 27.6364 45.6821 27.6364 43.8308C27.6364 41.9794 29.183 40.4786 31.0909 40.4786C32.9988 40.4786 34.5454 41.9794 34.5454 43.8308C34.5454 45.6821 32.9988 47.1829 31.0909 47.1829Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M44.9091 47.1829C43.0012 47.1829 41.4545 45.6821 41.4545 43.8308C41.4545 41.9794 43.0012 40.4786 44.9091 40.4786C46.817 40.4786 48.3636 41.9794 48.3636 43.8308C48.3636 45.6821 46.817 47.1829 44.9091 47.1829Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.7273 47.1829C56.8194 47.1829 55.2727 45.6821 55.2727 43.8308C55.2727 41.9794 56.8194 40.4786 58.7273 40.4786C60.6352 40.4786 62.1818 41.9794 62.1818 43.8308C62.1818 45.6821 60.6352 47.1829 58.7273 47.1829Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M72.5455 47.1829C70.6376 47.1829 69.0909 45.6821 69.0909 43.8308C69.0909 41.9794 70.6376 40.4786 72.5455 40.4786C74.4533 40.4786 76 41.9794 76 43.8308C76 45.6821 74.4533 47.1829 72.5455 47.1829Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M44.9091 60.5915C43.0012 60.5915 41.4545 59.0907 41.4545 57.2393C41.4545 55.388 43.0012 53.8872 44.9091 53.8872C46.817 53.8872 48.3636 55.388 48.3636 57.2393C48.3636 59.0907 46.817 60.5915 44.9091 60.5915Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M17.2727 47.1829C15.3648 47.1829 13.8182 45.6821 13.8182 43.8308C13.8182 41.9794 15.3648 40.4786 17.2727 40.4786C19.1806 40.4786 20.7273 41.9794 20.7273 43.8308C20.7273 45.6821 19.1806 47.1829 17.2727 47.1829Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M31.0909 74C29.183 74 27.6364 72.4992 27.6364 70.6479C27.6364 68.7965 29.183 67.2957 31.0909 67.2957C32.9988 67.2957 34.5455 68.7965 34.5455 70.6479C34.5455 72.4992 32.9988 74 31.0909 74Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M72.5455 74C70.6376 74 69.0909 72.4992 69.0909 70.6479C69.0909 68.7965 70.6376 67.2957 72.5455 67.2957C74.4533 67.2957 76 68.7965 76 70.6479C76 72.4992 74.4533 74 72.5455 74Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M3.45455 74C1.54665 74 1.2761e-06 72.4992 1.19369e-06 70.6479C1.11128e-06 68.7965 1.54665 67.2957 3.45455 67.2957C5.36244 67.2957 6.90909 68.7965 6.90909 70.6479C6.90909 72.4992 5.36244 74 3.45455 74Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                            </svg>

                            <svg class="slider-product-main-content-svg-two" width="99" height="47" viewBox="0 0 99 47" fill="none"
                                xmlns="http://www.w3.org/2000/svg">
                                <path
                                    d="M22.5 -5.51282C20.0147 -5.51282 18 -7.4598 18 -9.86153C18 -12.2633 20.0147 -14.2103 22.5 -14.2103C24.9853 -14.2103 27 -12.2633 27 -9.86153C27 -7.4598 24.9853 -5.51282 22.5 -5.51282Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.5 -5.51282C56.0147 -5.51282 54 -7.45981 54 -9.86154C54 -12.2633 56.0147 -14.2103 58.5 -14.2103C60.9853 -14.2103 63 -12.2633 63 -9.86154C63 -7.45981 60.9853 -5.51282 58.5 -5.51282Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M76.5 -5.51282C74.0147 -5.51282 72 -7.45981 72 -9.86154C72 -12.2633 74.0147 -14.2103 76.5 -14.2103C78.9853 -14.2103 81 -12.2633 81 -9.86154C81 -7.45981 78.9853 -5.51282 76.5 -5.51282Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M4.5 -22.9077C2.01472 -22.9077 8.84815e-07 -24.8547 7.77469e-07 -27.2564C6.70123e-07 -29.6581 2.01472 -31.6051 4.5 -31.6051C6.98528 -31.6051 9 -29.6581 9 -27.2564C9 -24.8547 6.98528 -22.9077 4.5 -22.9077Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M22.5 -22.9077C20.0147 -22.9077 18 -24.8547 18 -27.2564C18 -29.6581 20.0147 -31.6051 22.5 -31.6051C24.9853 -31.6051 27 -29.6581 27 -27.2564C27 -24.8547 24.9853 -22.9077 22.5 -22.9077Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M40.5 -22.9077C38.0147 -22.9077 36 -24.8547 36 -27.2564C36 -29.6581 38.0147 -31.6051 40.5 -31.6051C42.9853 -31.6051 45 -29.6581 45 -27.2564C45 -24.8547 42.9853 -22.9077 40.5 -22.9077Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M76.5 -22.9077C74.0147 -22.9077 72 -24.8547 72 -27.2564C72 -29.6581 74.0147 -31.6051 76.5 -31.6051C78.9853 -31.6051 81 -29.6581 81 -27.2564C81 -24.8547 78.9853 -22.9077 76.5 -22.9077Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M94.5 -22.9077C92.0147 -22.9077 90 -24.8547 90 -27.2564C90 -29.6581 92.0147 -31.6051 94.5 -31.6051C96.9853 -31.6051 99 -29.6581 99 -27.2564C99 -24.8547 96.9853 -22.9077 94.5 -22.9077Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M4.5 -40.3026C2.01472 -40.3026 1.07346e-07 -42.2495 0 -44.6513C-1.07346e-07 -47.053 2.01472 -49 4.5 -49C6.98528 -49 9 -47.053 9 -44.6513C9 -42.2495 6.98528 -40.3026 4.5 -40.3026Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M40.5 -40.3026C38.0147 -40.3026 36 -42.2496 36 -44.6513C36 -47.053 38.0147 -49 40.5 -49C42.9853 -49 45 -47.053 45 -44.6513C45 -42.2496 42.9853 -40.3026 40.5 -40.3026Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.5 -40.3026C56.0147 -40.3026 54 -42.2496 54 -44.6513C54 -47.053 56.0147 -49 58.5 -49C60.9853 -49 63 -47.053 63 -44.6513C63 -42.2496 60.9853 -40.3026 58.5 -40.3026Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M76.5 -40.3026C74.0147 -40.3026 72 -42.2496 72 -44.6513C72 -47.053 74.0147 -49 76.5 -49C78.9853 -49 81 -47.053 81 -44.6513C81 -42.2496 78.9853 -40.3026 76.5 -40.3026Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M94.5 -40.3026C92.0147 -40.3026 90 -42.2496 90 -44.6513C90 -47.053 92.0147 -49 94.5 -49C96.9853 -49 99 -47.053 99 -44.6513C99 -42.2496 96.9853 -40.3026 94.5 -40.3026Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.5 -22.9077C56.0147 -22.9077 54 -24.8547 54 -27.2564C54 -29.6581 56.0147 -31.6051 58.5 -31.6051C60.9853 -31.6051 63 -29.6581 63 -27.2564C63 -24.8547 60.9853 -22.9077 58.5 -22.9077Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M22.5 -40.3026C20.0147 -40.3026 18 -42.2495 18 -44.6513C18 -47.053 20.0147 -49 22.5 -49C24.9853 -49 27 -47.053 27 -44.6513C27 -42.2495 24.9853 -40.3026 22.5 -40.3026Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M40.5 -5.51282C38.0147 -5.51282 36 -7.4598 36 -9.86153C36 -12.2633 38.0147 -14.2103 40.5 -14.2103C42.9853 -14.2103 45 -12.2633 45 -9.86153C45 -7.4598 42.9853 -5.51282 40.5 -5.51282Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M94.5 -5.51282C92.0147 -5.51282 90 -7.45981 90 -9.86154C90 -12.2633 92.0147 -14.2103 94.5 -14.2103C96.9853 -14.2103 99 -12.2633 99 -9.86154C99 -7.45981 96.9853 -5.51282 94.5 -5.51282Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M4.5 -5.51282C2.01472 -5.51282 1.66229e-06 -7.4598 1.55494e-06 -9.86153C1.44759e-06 -12.2633 2.01472 -14.2103 4.5 -14.2103C6.98528 -14.2103 9 -12.2633 9 -9.86153C9 -7.4598 6.98528 -5.51282 4.5 -5.51282Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M22.5 47C20.0147 47 18 45.053 18 42.6513C18 40.2496 20.0147 38.3026 22.5 38.3026C24.9853 38.3026 27 40.2496 27 42.6513C27 45.053 24.9853 47 22.5 47Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.5 47C56.0147 47 54 45.053 54 42.6513C54 40.2496 56.0147 38.3026 58.5 38.3026C60.9853 38.3026 63 40.2496 63 42.6513C63 45.053 60.9853 47 58.5 47Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M76.5 47C74.0147 47 72 45.053 72 42.6513C72 40.2496 74.0147 38.3026 76.5 38.3026C78.9853 38.3026 81 40.2496 81 42.6513C81 45.053 78.9853 47 76.5 47Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M4.5 29.6051C2.01472 29.6051 8.84815e-07 27.6581 7.77469e-07 25.2564C6.70123e-07 22.8547 2.01472 20.9077 4.5 20.9077C6.98528 20.9077 9 22.8547 9 25.2564C9 27.6581 6.98528 29.6051 4.5 29.6051Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M22.5 29.6051C20.0147 29.6051 18 27.6581 18 25.2564C18 22.8547 20.0147 20.9077 22.5 20.9077C24.9853 20.9077 27 22.8547 27 25.2564C27 27.6581 24.9853 29.6051 22.5 29.6051Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M40.5 29.6051C38.0147 29.6051 36 27.6581 36 25.2564C36 22.8547 38.0147 20.9077 40.5 20.9077C42.9853 20.9077 45 22.8547 45 25.2564C45 27.6581 42.9853 29.6051 40.5 29.6051Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M76.5 29.6051C74.0147 29.6051 72 27.6581 72 25.2564C72 22.8547 74.0147 20.9077 76.5 20.9077C78.9853 20.9077 81 22.8547 81 25.2564C81 27.6581 78.9853 29.6051 76.5 29.6051Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M94.5 29.6051C92.0147 29.6051 90 27.6581 90 25.2564C90 22.8547 92.0147 20.9077 94.5 20.9077C96.9853 20.9077 99 22.8547 99 25.2564C99 27.6581 96.9853 29.6051 94.5 29.6051Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M4.5 12.2103C2.01472 12.2103 1.07346e-07 10.2633 0 7.86154C-1.07346e-07 5.45981 2.01472 3.51283 4.5 3.51283C6.98528 3.51283 9 5.45981 9 7.86154C9 10.2633 6.98528 12.2103 4.5 12.2103Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M40.5 12.2103C38.0147 12.2103 36 10.2633 36 7.86154C36 5.45981 38.0147 3.51283 40.5 3.51283C42.9853 3.51282 45 5.45981 45 7.86154C45 10.2633 42.9853 12.2103 40.5 12.2103Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.5 12.2103C56.0147 12.2103 54 10.2633 54 7.86154C54 5.45981 56.0147 3.51283 58.5 3.51283C60.9853 3.51282 63 5.45981 63 7.86154C63 10.2633 60.9853 12.2103 58.5 12.2103Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M76.5 12.2103C74.0147 12.2103 72 10.2633 72 7.86154C72 5.45981 74.0147 3.51283 76.5 3.51283C78.9853 3.51282 81 5.45981 81 7.86154C81 10.2633 78.9853 12.2103 76.5 12.2103Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M94.5 12.2103C92.0147 12.2103 90 10.2633 90 7.86154C90 5.45981 92.0147 3.51283 94.5 3.51283C96.9853 3.51282 99 5.45981 99 7.86154C99 10.2633 96.9853 12.2103 94.5 12.2103Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M58.5 29.6051C56.0147 29.6051 54 27.6581 54 25.2564C54 22.8547 56.0147 20.9077 58.5 20.9077C60.9853 20.9077 63 22.8547 63 25.2564C63 27.6581 60.9853 29.6051 58.5 29.6051Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M22.5 12.2103C20.0147 12.2103 18 10.2633 18 7.86154C18 5.45981 20.0147 3.51283 22.5 3.51283C24.9853 3.51282 27 5.45981 27 7.86154C27 10.2633 24.9853 12.2103 22.5 12.2103Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M40.5 47C38.0147 47 36 45.053 36 42.6513C36 40.2496 38.0147 38.3026 40.5 38.3026C42.9853 38.3026 45 40.2496 45 42.6513C45 45.053 42.9853 47 40.5 47Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M94.5 47C92.0147 47 90 45.053 90 42.6513C90 40.2496 92.0147 38.3026 94.5 38.3026C96.9853 38.3026 99 40.2496 99 42.6513C99 45.053 96.9853 47 94.5 47Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                                <path
                                    d="M4.5 47C2.01472 47 1.66229e-06 45.053 1.55494e-06 42.6513C1.44759e-06 40.2496 2.01472 38.3026 4.5 38.3026C6.98528 38.3026 9 40.2496 9 42.6513C9 45.053 6.98528 47 4.5 47Z"
                                    fill="#1D2977" fill-opacity="0.12" />
                            </svg>
                        </div>
                    </div>
                </div>
            <?php
            }
            ?>

            <a href="<?php echo esc_url($settings['all_categories_link']['url']) ?>"
                class="bottom-show-more-header-categories-main">
                <div>
                    <span><?php echo esc_html($settings['all_categories_text']) ?></span>
                </div>
                <div>
                    <svg width="40" height="40" viewBox="0 0 40 40" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M23.6216 13.6066C23.4648 13.7633 23.2504 13.8623 23.0029 13.8623L13.8623 13.8623L13.8623 23.0029C13.8623 23.4813 13.4664 23.8773 12.9879 23.8773C12.5094 23.8773 12.1134 23.4813 12.1134 23.0029L12.1134 12.9879C12.1134 12.5094 12.5094 12.1134 12.9879 12.1134L23.0029 12.1134C23.4813 12.1134 23.8773 12.5094 23.8773 12.9879C23.8856 13.2271 23.7783 13.4498 23.6216 13.6066Z"
                            fill="white" />
                        <path
                            d="M27.6308 27.6308C27.2926 27.969 26.7316 27.969 26.3934 27.6308L12.5093 13.7468C12.1711 13.4085 12.1711 12.8476 12.5093 12.5093C12.8476 12.1711 13.4085 12.1711 13.7468 12.5093L27.6308 26.3934C27.9691 26.7316 27.9691 27.2926 27.6308 27.6308Z"
                            fill="white" />
                    </svg>
                </div>
            </a>
        </section>

        <script>
            document.addEventListener("DOMContentLoaded", function() {
                const items = document.querySelectorAll(".slider-product-head-tab-item");
                if (items.length > 0) {
                    items[0].classList.add("active");
                }
                items.forEach(item => {
                    item.addEventListener("click", function() {
                        items.forEach(el => el.classList.remove("active"));
                        this.classList.add("active");
                    });
                });
            });

            document.addEventListener("DOMContentLoaded", function() {
                const tabs = document.querySelector(".slider-product-head-tabs");
                const items = tabs.querySelectorAll(".slider-product-head-tab-item");

                tabs.classList.add("fade");
            });
        </script>
<?php
    }
}
