<?php

namespace Elementor;

if (! defined('ABSPATH')) exit;

class Brand_Products extends Widget_Base
{
    public function get_name()
    {
        return 'Brand_Products';
    }

    public function get_title()
    {
        return __('برند های محصولات لمکده', 'lamkadeh');
    }

    public function get_icon()
    {
        return 'eicon-product-categories';
    }

    public function get_categories()
    {
        return ['lamkadeh'];
    }

    protected function register_controls()
    {
        // بخش تنظیمات محتوا
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('محتوا', 'lamkadeh'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'title',
            [
                'label' => __('عنوان', 'lamkadeh'),
                'type' => Controls_Manager::TEXT,
                'default' => __('محــبوب ترین <strong>بـــــــــرندها</strong>', 'lamkadeh'),
                'placeholder' => __('عنوان خود را وارد کنید', 'lamkadeh'),
            ]
        );

        $this->add_control(
            'description',
            [
                'label' => __('توضیحات', 'lamkadeh'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __('همکاران ما، <strong>بــــــازوان خلاقیت و کیفیت ما</strong> هستند. با همراهیه <strong>برندهای قابل اعتماد</strong>، خانه‌ای زیباتر می‌سازیم.', 'lamkadeh'),
                'placeholder' => __('توضیحات خود را وارد کنید', 'lamkadeh'),
            ]
        );

        $this->add_control(
            'number_of_brands',
            [
                'label' => __('تعداد برندها', 'lamkadeh'),
                'type' => Controls_Manager::NUMBER,
                'default' => 24,
                'min' => 1,
                'max' => 50,
            ]
        );

        $this->add_control(
            'is_enable_cache',
            [
                'label' => __('برند ها کش شوند؟ (پیشنهاد می شود فعال باشد.)', 'lamkadeh'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('بله', 'lamkadeh'),
                'label_off' => __('خیر', 'lamkadeh'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'time_cache',
            [
                'label' => __('زمان کش شدن برند ها', 'lamkadeh'),
                'type' => Controls_Manager::NUMBER,
                'default' => 24,
                'min' => 1,
                'max' => 72,
            ]
        );

        $this->end_controls_section();

        // بخش تنظیمات استایل
        $this->start_controls_section(
            'style_section',
            [
                'label' => __('استایل', 'lamkadeh'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __('رنگ عنوان', 'lamkadeh'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .title-header-categories-main span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label' => __('رنگ توضیحات', 'lamkadeh'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .description-header-categories-main' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $number_of_brands = $settings['number_of_brands'] ?? 24;
?>
        <section class="container">
            <section class="product-brands">
                <section class="header-categories-main">
                    <div class="title-header-categories-main">
                        <span>BEST SELLER PRODUCTS</span>
                        <div class="text-header-categories-main">
                            <span><?php echo wp_kses($settings['title'], ['strong' => []]); ?></span>
                            <svg width="64" height="64" viewBox="0 0 64 64" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <g clip-path="url(#clip0_1_6993)">
                                    <circle cx="32" cy="32" r="32" transform="matrix(-1 0 0 1 64 0)" fill="#FFD701" />
                                    <path fill-rule="evenodd" clip-rule="evenodd" d="M49.817 20.5089C49.4772 19.9205 48.8545 19.727 48.2601 19.6439C47.6609 19.5601 46.8589 19.5601 45.8889 19.5602L45.8337 19.5602C45.2561 19.5602 44.8784 19.5545 44.6033 19.5162C44.3545 19.4816 44.2754 19.4304 44.2416 19.4006C44.2169 19.3748 44.1729 19.3145 44.1429 19.0973C44.1075 18.8412 44.106 18.49 44.106 17.9271V17.5838C44.106 16.1286 44.106 14.9628 43.994 14.1317C43.8889 13.3515 43.639 12.4753 42.7735 12.2101C41.9232 11.9495 41.21 12.5083 40.6556 13.0843C40.0651 13.698 39.3636 14.6472 38.4853 15.8356L35.6097 19.7262C35.0502 20.4831 34.5838 21.1141 34.3059 21.6358C34.0274 22.1588 33.8246 22.7798 34.1336 23.3993L34.1347 23.4016L34.138 23.4083L34.1417 23.4156L34.1453 23.4226L34.1493 23.4301L34.1533 23.4376L34.1572 23.4445L34.1612 23.4516L34.1649 23.4579L34.1663 23.4602C34.5032 24.0436 35.1232 24.2479 35.723 24.3394C36.3411 24.4338 37.1659 24.4399 38.1663 24.4399C38.7498 24.4399 39.1219 24.4412 39.3952 24.476C39.6366 24.5068 39.7088 24.5542 39.7397 24.5816C39.7671 24.6102 39.8152 24.6771 39.8491 24.9022C39.8879 25.1598 39.894 25.5165 39.894 26.073L39.894 26.4161C39.8939 27.8714 39.8939 29.0371 40.0059 29.8683C40.111 30.6484 40.3609 31.5246 41.2264 31.7899C42.0767 32.0505 42.7899 31.4917 43.3442 30.9157C43.9348 30.302 44.6363 29.3528 45.5147 28.1644L48.3558 24.3205C48.934 23.5382 49.4095 22.8846 49.69 22.3406C49.9644 21.8083 50.1559 21.1837 49.8498 20.5698L49.8487 20.5676L49.8454 20.5609L49.8417 20.5536L49.8381 20.5465L49.8341 20.539L49.83 20.5315L49.8261 20.5246L49.8221 20.5175L49.8183 20.5111L49.817 20.5089ZM41.6464 14.0378C41.1258 14.5788 40.477 15.4541 39.5569 16.699L36.7499 20.4968C36.147 21.3125 35.7437 21.8613 35.5195 22.2822C35.4104 22.487 35.3695 22.6162 35.3578 22.6945C35.3506 22.7428 35.355 22.7646 35.3608 22.7786C35.3829 22.8112 35.4792 22.9113 35.9305 22.9802C36.4236 23.0555 37.1326 23.0649 38.1663 23.0649L38.2084 23.0649C38.7374 23.0648 39.1984 23.0648 39.5691 23.1121C39.9662 23.1627 40.3606 23.2778 40.6872 23.585L40.6925 23.59L40.6977 23.5951C41.0186 23.9101 41.149 24.3005 41.2088 24.6975C41.2659 25.0768 41.269 25.5435 41.269 26.073L41.269 26.3568C41.269 27.8848 41.2705 28.9568 41.3686 29.6847C41.4174 30.0471 41.4831 30.2597 41.546 30.3785C41.5931 30.4674 41.6203 30.4733 41.6279 30.4749L41.6293 30.4752L41.6307 30.4757C41.6403 30.479 41.674 30.4907 41.7782 30.4382C41.9084 30.3727 42.0944 30.2315 42.3535 29.9622C42.8741 29.4212 43.5229 28.5459 44.443 27.301L47.25 23.5032C47.8495 22.6922 48.2489 22.1353 48.4679 21.7106C48.6555 21.3465 48.6351 21.2232 48.6226 21.1905C48.602 21.1604 48.5096 21.0672 48.0697 21.0056C47.5773 20.9368 46.8728 20.9352 45.8337 20.9352C45.2811 20.9352 44.8018 20.932 44.414 20.8781C44.0135 20.8224 43.6182 20.7019 43.2944 20.3974L43.2891 20.3924L43.2839 20.3873C42.9601 20.0695 42.8352 19.6795 42.7808 19.2854C42.7309 18.924 42.7309 18.4763 42.731 17.9714L42.731 17.6432C42.731 16.1152 42.7294 15.0432 42.6313 14.3153C42.5825 13.9529 42.5168 13.7403 42.4539 13.6215C42.4067 13.5326 42.3796 13.5267 42.372 13.5251L42.3706 13.5248L42.3692 13.5243C42.3596 13.521 42.3258 13.5093 42.2216 13.5618C42.0915 13.6273 41.9055 13.7685 41.6464 14.0378Z" fill="#1D2977" />
                                    <circle cx="22" cy="60" r="17" fill="#F7F8FD" fill-opacity="0.64" />
                                </g>
                                <defs>
                                    <clipPath id="clip0_1_6993">
                                        <rect width="64" height="64" rx="32" fill="white" />
                                    </clipPath>
                                </defs>
                            </svg>
                        </div>
                        <div class="shape-header-categories-main"></div>
                    </div>
                    <p class="description-header-categories-main">
                        <?php echo wp_kses($settings['description'], ['strong' => []]); ?>
                    </p>
                </section>

                <div class="product-brands-list">
                    <?php
                    if ($settings['is_enable_cache'] === "yes") {
                        $brands = get_transient('cached_brands');
                        if (false === $brands) {
                            $brands = get_terms(array(
                                'taxonomy' => 'product_brand',
                                'hide_empty' => true,
                                'number' => $number_of_brands,
                                'no_found_rows'          => true,
                                'update_post_meta_cache' => false,
                                'update_post_term_cache' => false,
                            ));
                            set_transient('cached_brands', $brands, (int) $settings['time_cache'] * HOUR_IN_SECONDS);
                        }
                    } else {
                        $brands = get_terms(array(
                            'taxonomy' => 'product_brand',
                            'hide_empty' => true,
                            'number' => $number_of_brands,
                            'no_found_rows'          => true,
                            'update_post_meta_cache' => false,
                            'update_post_term_cache' => false,
                        ));
                    }

                    if (!empty($brands) && !is_wp_error($brands)) {
                        $first_brand = $brands[0];
                        function get_product_prices($product)
                        {
                            $prices = array(
                                'regular_price' => '',
                                'sale_price' => '',
                                'price' => '',
                                'is_on_sale' => false,
                                'discount_percentage' => 0
                            );

                            if ($product->is_type('variable')) {
                                $variation_prices = $product->get_variation_prices(true);

                                if (!empty($variation_prices['price'])) {
                                    $min_regular_price = min($variation_prices['regular_price']);
                                    $min_price = min($variation_prices['price']);

                                    $prices['regular_price'] = $min_regular_price;
                                    $prices['price'] = $min_price;
                                    $prices['sale_price'] = $min_price;
                                    $prices['is_on_sale'] = $min_price < $min_regular_price;

                                    if ($prices['is_on_sale'] && $min_regular_price > 0) {
                                        $prices['discount_percentage'] = round(($min_regular_price - $min_price) / $min_regular_price * 100);
                                    }
                                }
                            } else {
                                $prices['regular_price'] = $product->get_regular_price();
                                $prices['sale_price'] = $product->get_sale_price();
                                $prices['price'] = $product->get_price();
                                $prices['is_on_sale'] = $product->is_on_sale();

                                if ($prices['is_on_sale'] && $prices['regular_price'] > 0) {
                                    $prices['discount_percentage'] = round(($prices['regular_price'] - $prices['sale_price']) / $prices['regular_price'] * 100);
                                }
                            }

                            return $prices;
                        }

                        function get_product_category($product_id)
                        {
                            $categories = get_the_terms($product_id, 'product_cat');
                            if ($categories && !is_wp_error($categories)) {
                                return $categories[0];
                            }
                            return null;
                        }

                        function format_price_without_symbol($price)
                        {
                            if (empty($price)) return '';

                            $price = floatval($price);
                            $formatted = number_format($price);

                            if (floor($price) == $price) {
                                return $formatted;
                            } else {
                                return number_format($price, 0, '.', ',');
                            }
                        }

                        function get_brand_products($brand_id, $posts_per_page = 4)
                        {
                            $args = array(
                                'post_type' => 'product',
                                'posts_per_page' => $posts_per_page,
                                'tax_query' => array(
                                    array(
                                        'taxonomy' => 'product_brand',
                                        'field' => 'term_id',
                                        'terms' => $brand_id,
                                    )
                                ),
                                'meta_key' => 'total_sales',
                                'orderby' => 'meta_value_num',
                                'order' => 'DESC',
                                'no_found_rows'          => true,
                                'update_post_meta_cache' => false,
                                'update_post_term_cache' => false,
                            );
                            return get_posts($args);
                        }

                        // تابع برای نمایش محصولات
                        function display_products($products, $brand_id)
                        {
                            if ($products) {
                                $product_chunks = array_chunk($products, 2);

                                foreach ($product_chunks as $chunk_index => $product_chunk): ?>
                                    <div class="product-brands-list-left-bottom-items" data-brand-id="<?php echo $brand_id; ?>">
                                        <?php foreach ($product_chunk as $product):
                                            $product_id = $product->ID;
                                            $wc_product = wc_get_product($product_id);

                                            if (!$wc_product) continue;

                                            $product_prices = get_product_prices($wc_product);
                                            $product_category = get_product_category($product_id);
                                        ?>
                                            <div class="product-brands-list-left-bottom-item">
                                                <div class="product-brands-list-left-bottom-item-image">
                                                    <?php
                                                    $product_image = get_the_post_thumbnail_url($product_id, 'medium');
                                                    $fallback_image = 'http://lamkadeh.com/wp-content/uploads/2025/08/6ab22bedd099068b507027f9225221324b2b9b85-300x300.png';
                                                    ?>
                                                    <img src="<?php echo esc_url($product_image ?: $fallback_image); ?>" alt="<?php echo esc_attr(get_the_title($product_id)); ?>">
                                                </div>
                                                <div class="product-brands-list-left-bottom-item-content">
                                                    <div class="product-brands-list-left-bottom-item-content-header">
                                                        <a href="<?php echo esc_url(get_permalink($product_id)); ?>"><?php echo esc_html(get_the_title($product_id)); ?></a>
                                                        <div class="product-brands-list-left-bottom-item-content-header-category">
                                                            <?php if ($product_category): ?>
                                                                <a href="<?php echo esc_url(get_term_link($product_category)); ?>" class="product-brands-list-left-bottom-item-content-header-category-item">
                                                                    <span><?php echo esc_html($product_category->name); ?></span>
                                                                    <?php
                                                                    $category_image = get_term_meta($product_category->term_id, 'thumbnail_id', true);
                                                                    $category_image_url = $category_image ? wp_get_attachment_image_url($category_image, 'thumbnail') : 'http://lamkadeh.com/wp-content/uploads/2025/08/chair-1-150x150.png';
                                                                    ?>
                                                                    <img src="<?php echo esc_url($category_image_url); ?>" alt="" />
                                                                </a>
                                                            <?php endif; ?>
                                                            <div class="product-brands-list-left-bottom-item-content-header-category-item">
                                                                <span><?php echo get_comments_number($product_id); ?> نظر</span>
                                                                <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                    <g clip-path="url(#clip0_1_7131)">
                                                                        <path fill-rule="evenodd" clip-rule="evenodd" d="M6.74174 3.98083C7.62671 2.17909 9.48027 0.9375 11.625 0.9375C14.628 0.9375 17.0625 3.37195 17.0625 6.375C17.0625 7.17907 16.8876 7.94382 16.5733 8.632C16.4912 8.81161 16.4753 8.98125 16.5117 9.11757L16.6073 9.4746C16.913 10.6175 15.8675 11.663 14.7246 11.3573L14.3676 11.2617C14.2661 11.2346 14.1462 11.2365 14.0174 11.2731C13.6349 14.5331 10.8628 17.0625 7.5 17.0625C6.48752 17.0625 5.52711 16.8328 4.66942 16.4222C4.49963 16.3409 4.33245 16.3244 4.19493 16.3612L3.2754 16.6073C2.13253 16.913 1.08695 15.8675 1.39275 14.7246L1.63878 13.8051C1.67558 13.6676 1.65908 13.5004 1.5778 13.3306C1.16718 12.4729 0.9375 11.5125 0.9375 10.5C0.9375 7.13209 3.47455 4.35673 6.74174 3.98083ZM8.05124 3.96033C11.2948 4.23009 13.8686 6.85883 14.052 10.1262C14.2489 10.1074 14.4536 10.1202 14.6584 10.175L15.0154 10.2705C15.322 10.3525 15.6025 10.072 15.5205 9.76538L15.425 9.40835C15.3071 8.96785 15.3839 8.52811 15.55 8.16458C15.7986 7.62016 15.9375 7.01456 15.9375 6.375C15.9375 3.99327 14.0067 2.0625 11.625 2.0625C10.1386 2.0625 8.8269 2.81463 8.05124 3.96033ZM7.5 5.0625C4.49695 5.0625 2.0625 7.49695 2.0625 10.5C2.0625 11.3408 2.25294 12.1355 2.5925 12.8448C2.76906 13.2136 2.84309 13.6566 2.72555 14.0959L2.47952 15.0154C2.39748 15.322 2.678 15.6025 2.98462 15.5205L3.90415 15.2744C4.34345 15.1569 4.78643 15.2309 5.15521 15.4075C5.86449 15.7471 6.65922 15.9375 7.5 15.9375C10.503 15.9375 12.9375 13.503 12.9375 10.5C12.9375 7.49695 10.503 5.0625 7.5 5.0625Z" fill="#020308" fill-opacity="0.64" />
                                                                        <path d="M5.625 10.5C5.625 10.9142 5.28921 11.25 4.875 11.25C4.46079 11.25 4.125 10.9142 4.125 10.5C4.125 10.0858 4.46079 9.75 4.875 9.75C5.28921 9.75 5.625 10.0858 5.625 10.5Z" fill="#020308" fill-opacity="0.64" />
                                                                        <path d="M8.25 10.5C8.25 10.9142 7.91421 11.25 7.5 11.25C7.08579 11.25 6.75 10.9142 6.75 10.5C6.75 10.0858 7.08579 9.75 7.5 9.75C7.91421 9.75 8.25 10.0858 8.25 10.5Z" fill="#020308" fill-opacity="0.64" />
                                                                        <path d="M10.875 10.5C10.875 10.9142 10.5392 11.25 10.125 11.25C9.71079 11.25 9.375 10.9142 9.375 10.5C9.375 10.0858 9.71079 9.75 10.125 9.75C10.5392 9.75 10.875 10.0858 10.875 10.5Z" fill="#020308" fill-opacity="0.64" />
                                                                    </g>
                                                                    <defs>
                                                                        <clipPath id="clip0_1_7131">
                                                                            <rect width="18" height="18" fill="white" />
                                                                        </clipPath>
                                                                    </defs>
                                                                </svg>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <div class="underline-lamkadeh">
                                                        <div class="underline-lamkadeh-line"></div>
                                                        <div class="underline-lamkadeh-circle"></div>
                                                    </div>
                                                    <div class="product-brands-list-left-bottom-item-content-footer">
                                                        <div class="product-brands-list-left-bottom-item-content-footer-item">
                                                            <?php if ($product_prices['is_on_sale'] && $product_prices['regular_price'] > 0 && $product_prices['price'] < $product_prices['regular_price']): ?>
                                                                <div class="product-brands-list-left-bottom-item-content-footer-item-regular-price">
                                                                    <span><?php echo format_price_without_symbol($product_prices['regular_price']); ?></span>
                                                                </div>
                                                                <div class="product-brands-list-left-bottom-item-content-footer-item-line"></div>
                                                                <div class="product-brands-list-left-bottom-item-content-footer-item-salesoff-price">
                                                                    <div class="product-brands-list-left-bottom-item-content-footer-item-salesoff-price-rectangle"></div>
                                                                    <span><?php echo format_price_without_symbol($product_prices['price']); ?></span>
                                                                    <div class="product-brands-list-left-bottom-item-content-footer-item-salesoff-price-symbol">تــــو مان</div>
                                                                </div>
                                                            <?php else: ?>
                                                                <div class="product-brands-list-left-bottom-item-content-footer-item-salesoff-price">
                                                                    <span><?php echo format_price_without_symbol($product_prices['price']); ?></span>
                                                                    <div class="product-brands-list-left-bottom-item-content-footer-item-salesoff-price-symbol">تــــو مان</div>
                                                                </div>
                                                            <?php endif; ?>
                                                        </div>
                                                        <div class="product-brands-list-left-bottom-item-content-footer-item">
                                                            <?php if (is_user_logged_in()) :
                                                                $user_id = get_current_user_id();
                                                                $wishlist = get_user_meta($user_id, 'lamkadeh_wishlist', true);
                                                                $is_favorite = is_array($wishlist) && in_array($wc_product->get_id(), $wishlist);
                                                            ?>
                                                                <div class="wishlist-btn product-brands-list-left-bottom-item-content-footer-item-icon <?php echo $is_favorite ? 'active' : ''; ?>" data-product-id="<?php echo esc_attr($wc_product->get_id()); ?>">
                                                                    <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                        <path fill-rule="evenodd" clip-rule="evenodd" d="M4.21827 3.31808C2.97403 3.88683 2.0625 5.23961 2.0625 6.85276C2.0625 8.50081 2.73691 9.77111 3.70371 10.8598C4.50054 11.757 5.46513 12.5007 6.40584 13.2259C6.62928 13.3981 6.85136 13.5694 7.06953 13.7413C7.46406 14.0524 7.81599 14.3253 8.15521 14.5236C8.49461 14.7219 8.7678 14.8124 9 14.8124C9.2322 14.8124 9.50539 14.7219 9.84479 14.5236C10.184 14.3253 10.5359 14.0524 10.9305 13.7413C11.1486 13.5694 11.3707 13.3981 11.5942 13.2259C12.5349 12.5007 13.4995 11.757 14.2963 10.8598C15.2631 9.77111 15.9375 8.50081 15.9375 6.85276C15.9375 5.23961 15.026 3.88683 13.7817 3.31808C12.5729 2.76554 10.9488 2.91187 9.40527 4.51548C9.29923 4.62566 9.15291 4.68791 9 4.68791C8.84709 4.68791 8.70077 4.62566 8.59473 4.51548C7.05125 2.91187 5.42705 2.76554 4.21827 3.31808ZM9 3.34404C7.26596 1.79261 5.32422 1.57559 3.75057 2.2949C2.08853 3.05463 0.9375 4.81871 0.9375 6.85276C0.9375 8.85191 1.77037 10.377 2.86254 11.6068C3.73715 12.5916 4.80767 13.4159 5.75312 14.1439C5.96744 14.3089 6.17533 14.469 6.37306 14.6248C6.75724 14.9277 7.16966 15.2506 7.58762 15.4949C8.00539 15.739 8.4822 15.9374 9 15.9374C9.5178 15.9374 9.99461 15.739 10.4124 15.4949C10.8303 15.2506 11.2428 14.9277 11.6269 14.6248C11.8247 14.469 12.0326 14.3089 12.2469 14.1439C13.1923 13.4159 14.2628 12.5916 15.1375 11.6068C16.2296 10.377 17.0625 8.85191 17.0625 6.85276C17.0625 4.81871 15.9115 3.05463 14.2494 2.2949C12.6758 1.57559 10.734 1.79261 9 3.34404Z" fill="#1D2977" />
                                                                    </svg>
                                                                </div>
                                                            <?php else : ?>
                                                                <div class="wishlist-btn product-brands-list-left-bottom-item-content-footer-item-icon" onclick="loginBeforeWhishlist()">
                                                                    <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                        <path fill-rule="evenodd" clip-rule="evenodd" d="M4.21827 3.31808C2.97403 3.88683 2.0625 5.23961 2.0625 6.85276C2.0625 8.50081 2.73691 9.77111 3.70371 10.8598C4.50054 11.757 5.46513 12.5007 6.40584 13.2259C6.62928 13.3981 6.85136 13.5694 7.06953 13.7413C7.46406 14.0524 7.81599 14.3253 8.15521 14.5236C8.49461 14.7219 8.7678 14.8124 9 14.8124C9.2322 14.8124 9.50539 14.7219 9.84479 14.5236C10.184 14.3253 10.5359 14.0524 10.9305 13.7413C11.1486 13.5694 11.3707 13.3981 11.5942 13.2259C12.5349 12.5007 13.4995 11.757 14.2963 10.8598C15.2631 9.77111 15.9375 8.50081 15.9375 6.85276C15.9375 5.23961 15.026 3.88683 13.7817 3.31808C12.5729 2.76554 10.9488 2.91187 9.40527 4.51548C9.29923 4.62566 9.15291 4.68791 9 4.68791C8.84709 4.68791 8.70077 4.62566 8.59473 4.51548C7.05125 2.91187 5.42705 2.76554 4.21827 3.31808ZM9 3.34404C7.26596 1.79261 5.32422 1.57559 3.75057 2.2949C2.08853 3.05463 0.9375 4.81871 0.9375 6.85276C0.9375 8.85191 1.77037 10.377 2.86254 11.6068C3.73715 12.5916 4.80767 13.4159 5.75312 14.1439C5.96744 14.3089 6.17533 14.469 6.37306 14.6248C6.75724 14.9277 7.16966 15.2506 7.58762 15.4949C8.00539 15.739 8.4822 15.9374 9 15.9374C9.5178 15.9374 9.99461 15.739 10.4124 15.4949C10.8303 15.2506 11.2428 14.9277 11.6269 14.6248C11.8247 14.469 12.0326 14.3089 12.2469 14.1439C13.1923 13.4159 14.2628 12.5916 15.1375 11.6068C16.2296 10.377 17.0625 8.85191 17.0625 6.85276C17.0625 4.81871 15.9115 3.05463 14.2494 2.2949C12.6758 1.57559 10.734 1.79261 9 3.34404Z" fill="#1D2977" />
                                                                    </svg>
                                                                </div>
                                                            <?php endif; ?>
                                                            <a href="<?php echo esc_url(get_permalink($product_id)); ?>" class="product-brands-list-left-bottom-item-content-footer-item-icon">
                                                                <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                    <path fill-rule="evenodd" clip-rule="evenodd" d="M9.00017 2.0625C8.06819 2.0625 7.31267 2.81802 7.31267 3.75V3.94086C7.62579 3.93749 7.96382 3.9375 8.3291 3.9375L9.67124 3.9375C10.0365 3.9375 10.3745 3.93749 10.6877 3.94086V3.75C10.6877 2.81802 9.93215 2.0625 9.00017 2.0625ZM11.8127 3.9802V3.75C11.8127 2.1967 10.5535 0.9375 9.00017 0.9375C7.44687 0.9375 6.18767 2.1967 6.18767 3.75V3.9802C6.08642 3.98759 5.98872 3.99626 5.8944 4.00642C5.23563 4.07742 4.68473 4.22616 4.19551 4.56148C4.02725 4.67681 3.8692 4.80637 3.72312 4.94875C3.29837 5.36271 3.04448 5.87374 2.84566 6.50579C2.65263 7.11944 2.49662 7.8995 2.30059 8.87969L2.28628 8.95123C2.0038 10.3636 1.78121 11.4765 1.71728 12.363C1.65195 13.2687 1.74324 14.0306 2.17226 14.691C2.31541 14.9113 2.48249 15.1151 2.67045 15.2987C3.2338 15.8488 3.96306 16.0878 4.86401 16.2014C5.74577 16.3125 6.88075 16.3125 8.32107 16.3125H9.67927C11.1196 16.3125 12.2546 16.3125 13.1363 16.2014C14.0373 16.0878 14.7665 15.8488 15.3299 15.2987C15.5179 15.1151 15.6849 14.9113 15.8281 14.691C16.2571 14.0306 16.3484 13.2687 16.2831 12.363C16.2191 11.4765 15.9965 10.3636 15.7141 8.95129L15.6998 8.87972C15.5037 7.89951 15.3477 7.11944 15.1547 6.50579C14.9559 5.87374 14.702 5.36271 14.2772 4.94875C14.1311 4.80637 13.9731 4.6768 13.8048 4.56148C13.3156 4.22616 12.7647 4.07742 12.1059 4.00642C12.0116 3.99626 11.9139 3.98759 11.8127 3.9802ZM6.01496 5.12494C5.45493 5.1853 5.10895 5.29929 4.83153 5.48943C4.71641 5.56834 4.60827 5.65699 4.50832 5.75441C4.26746 5.98914 4.08784 6.30605 3.91882 6.84337C3.74609 7.39246 3.60122 8.11294 3.39738 9.13216C3.10522 10.5929 2.89759 11.6365 2.83936 12.4439C2.78192 13.2403 2.87891 13.7137 3.11563 14.078C3.21357 14.2288 3.32789 14.3682 3.45649 14.4938C3.76732 14.7974 4.21247 14.9853 5.00471 15.0852C5.80788 15.1864 6.87185 15.1875 8.36157 15.1875H9.63877C11.1285 15.1875 12.1925 15.1864 12.9956 15.0852C13.7879 14.9853 14.233 14.7974 14.5439 14.4938C14.6725 14.3682 14.7868 14.2288 14.8847 14.078C15.1214 13.7137 15.2184 13.2403 15.161 12.4439C15.1027 11.6365 14.8951 10.5929 14.603 9.13216C14.3991 8.11294 14.2542 7.39246 14.0815 6.84337C13.9125 6.30605 13.7329 5.98914 13.492 5.75441C13.3921 5.65699 13.2839 5.56834 13.1688 5.48943C12.8914 5.29929 12.5454 5.1853 11.9854 5.12494C11.4131 5.06326 10.6782 5.0625 9.63877 5.0625H8.36157C7.32216 5.0625 6.58726 5.06326 6.01496 5.12494ZM6.69089 10.7197C6.9838 10.6161 7.30517 10.7696 7.40869 11.0625C7.64062 11.7187 8.26645 12.1875 9.00032 12.1875C9.73418 12.1875 10.36 11.7187 10.5919 11.0625C10.6955 10.7696 11.0168 10.6161 11.3097 10.7197C11.6026 10.8232 11.7562 11.1445 11.6526 11.4375C11.2668 12.5291 10.2258 13.3125 9.00032 13.3125C7.77486 13.3125 6.73383 12.5291 6.348 11.4375C6.24447 11.1445 6.39799 10.8232 6.69089 10.7197Z" fill="#1D2977" />
                                                                </svg>
                                                            </a>
                                                        </div>
                                                    </div>
                                                </div>
                                                <?php if ($product_prices['discount_percentage'] > 0): ?>
                                                    <div class="product-brands-list-left-bottom-item-badge">
                                                        <span>%<?php echo esc_html($product_prices['discount_percentage']); ?></span>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                        <?php endforeach;
                            } else {
                                echo '<div class="no-products">هیچ محصولی برای این برند یافت نشد.</div>';
                            }
                        }
                        ?>

                        <div class="product-brands-list-right container">
                            <div class="product-brands-list-right-header">
                                <?php
                                $brand_image = get_term_meta($first_brand->term_id, 'thumbnail_id', true);
                                $image_url = $brand_image ? wp_get_attachment_url($brand_image) : 'https://pngdl.ir/wp-content/uploads/2024/07/258851.jpg';
                                ?>
                                <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($first_brand->name); ?>">
                                <div class="product-brands-list-right-header-button">
                                    <?php
                                    $product_count = $first_brand->count;
                                    ?>
                                    <span><strong><?php echo number_format($product_count); ?>+</strong> <?php echo __("محصول", "lamkadeh") ?></span>
                                </div>
                                <svg class="rectangle-svg" width="65" height="41" viewBox="0 0 65 41" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M14.7727 28.0855C13.141 28.0855 11.8182 26.828 11.8182 25.2769C11.8182 23.7258 13.141 22.4684 14.7727 22.4684C16.4045 22.4684 17.7273 23.7258 17.7273 25.2769C17.7273 26.828 16.4045 28.0855 14.7727 28.0855Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M38.4091 28.0855C36.7773 28.0855 35.4545 26.828 35.4545 25.2769C35.4545 23.7258 36.7773 22.4684 38.4091 22.4684C40.0408 22.4684 41.3636 23.7258 41.3636 25.2769C41.3636 26.828 40.0408 28.0855 38.4091 28.0855Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M50.2273 28.0855C48.5955 28.0855 47.2727 26.828 47.2727 25.2769C47.2727 23.7258 48.5955 22.4684 50.2273 22.4684C51.859 22.4684 53.1818 23.7258 53.1818 25.2769C53.1818 26.828 51.859 28.0855 50.2273 28.0855Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M2.95455 16.8513C1.3228 16.8513 5.80939e-07 15.5939 5.1046e-07 14.0427C4.3998e-07 12.4916 1.3228 11.2342 2.95455 11.2342C4.5863 11.2342 5.90909 12.4916 5.90909 14.0427C5.90909 15.5939 4.5863 16.8513 2.95455 16.8513Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M14.7727 16.8513C13.141 16.8513 11.8182 15.5939 11.8182 14.0427C11.8182 12.4916 13.141 11.2342 14.7727 11.2342C16.4045 11.2342 17.7273 12.4916 17.7273 14.0427C17.7273 15.5939 16.4045 16.8513 14.7727 16.8513Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M26.5909 16.8513C24.9592 16.8513 23.6364 15.5939 23.6364 14.0427C23.6364 12.4916 24.9592 11.2342 26.5909 11.2342C28.2227 11.2342 29.5455 12.4916 29.5455 14.0427C29.5455 15.5939 28.2227 16.8513 26.5909 16.8513Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M50.2273 16.8513C48.5955 16.8513 47.2727 15.5939 47.2727 14.0427C47.2727 12.4916 48.5955 11.2342 50.2273 11.2342C51.859 11.2342 53.1818 12.4916 53.1818 14.0427C53.1818 15.5939 51.859 16.8513 50.2273 16.8513Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M62.0455 16.8513C60.4137 16.8513 59.0909 15.5939 59.0909 14.0427C59.0909 12.4916 60.4137 11.2342 62.0455 11.2342C63.6772 11.2342 65 12.4916 65 14.0427C65 15.5939 63.6772 16.8513 62.0455 16.8513Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M2.95455 5.61709C1.3228 5.61709 7.04798e-08 4.35967 0 2.80855C-7.04798e-08 1.25743 1.32279 2.55341e-06 2.95455 2.48479e-06C4.5863 2.41617e-06 5.90909 1.25743 5.90909 2.80855C5.90909 4.35967 4.5863 5.61709 2.95455 5.61709Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M26.5909 5.61709C24.9592 5.61709 23.6364 4.35967 23.6364 2.80855C23.6364 1.25743 24.9592 1.55949e-06 26.5909 1.49087e-06C28.2227 1.42226e-06 29.5455 1.25743 29.5455 2.80855C29.5455 4.35967 28.2227 5.61709 26.5909 5.61709Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M38.4091 5.61709C36.7773 5.61709 35.4545 4.35966 35.4545 2.80855C35.4545 1.25743 36.7773 1.06253e-06 38.4091 9.93916e-07C40.0408 9.25301e-07 41.3636 1.25743 41.3636 2.80855C41.3636 4.35966 40.0408 5.61709 38.4091 5.61709Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M50.2273 5.61709C48.5955 5.61709 47.2727 4.35966 47.2727 2.80855C47.2727 1.25743 48.5955 5.65573e-07 50.2273 4.96958e-07C51.859 4.28342e-07 53.1818 1.25743 53.1818 2.80855C53.1818 4.35966 51.859 5.61709 50.2273 5.61709Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M62.0455 5.61709C60.4137 5.61709 59.0909 4.35966 59.0909 2.80855C59.0909 1.25743 60.4137 6.86156e-08 62.0455 0C63.6772 -6.86156e-08 65 1.25743 65 2.80855C65 4.35966 63.6772 5.61709 62.0455 5.61709Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M38.4091 16.8513C36.7773 16.8513 35.4545 15.5939 35.4545 14.0427C35.4545 12.4916 36.7773 11.2342 38.4091 11.2342C40.0408 11.2342 41.3636 12.4916 41.3636 14.0427C41.3636 15.5939 40.0408 16.8513 38.4091 16.8513Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M14.7727 5.61709C13.141 5.61709 11.8182 4.35967 11.8182 2.80855C11.8182 1.25743 13.141 2.05645e-06 14.7727 1.98783e-06C16.4045 1.91922e-06 17.7273 1.25743 17.7273 2.80855C17.7273 4.35967 16.4045 5.61709 14.7727 5.61709Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M26.5909 28.0855C24.9592 28.0855 23.6364 26.828 23.6364 25.2769C23.6364 23.7258 24.9592 22.4684 26.5909 22.4684C28.2227 22.4684 29.5455 23.7258 29.5455 25.2769C29.5455 26.828 28.2227 28.0855 26.5909 28.0855Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M62.0455 28.0855C60.4137 28.0855 59.0909 26.828 59.0909 25.2769C59.0909 23.7258 60.4137 22.4684 62.0455 22.4684C63.6772 22.4684 65 23.7258 65 25.2769C65 26.828 63.6772 28.0855 62.0455 28.0855Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M2.95455 28.0855C1.3228 28.0855 1.0914e-06 26.828 1.02092e-06 25.2769C9.5044e-07 23.7258 1.3228 22.4684 2.95455 22.4684C4.5863 22.4684 5.90909 23.7258 5.90909 25.2769C5.90909 26.828 4.5863 28.0855 2.95455 28.0855Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M14.7727 62C13.141 62 11.8182 60.7426 11.8182 59.1915C11.8182 57.6403 13.141 56.3829 14.7727 56.3829C16.4045 56.3829 17.7273 57.6403 17.7273 59.1915C17.7273 60.7426 16.4045 62 14.7727 62Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M38.4091 62C36.7773 62 35.4545 60.7426 35.4545 59.1915C35.4545 57.6403 36.7773 56.3829 38.4091 56.3829C40.0408 56.3829 41.3636 57.6403 41.3636 59.1915C41.3636 60.7426 40.0408 62 38.4091 62Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M50.2273 62C48.5955 62 47.2727 60.7426 47.2727 59.1915C47.2727 57.6403 48.5955 56.3829 50.2273 56.3829C51.859 56.3829 53.1818 57.6403 53.1818 59.1915C53.1818 60.7426 51.859 62 50.2273 62Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M2.95455 50.7658C1.3228 50.7658 5.80939e-07 49.5084 5.1046e-07 47.9573C4.3998e-07 46.4062 1.3228 45.1487 2.95455 45.1487C4.5863 45.1487 5.90909 46.4062 5.90909 47.9573C5.90909 49.5084 4.5863 50.7658 2.95455 50.7658Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M14.7727 50.7658C13.141 50.7658 11.8182 49.5084 11.8182 47.9573C11.8182 46.4062 13.141 45.1487 14.7727 45.1487C16.4045 45.1487 17.7273 46.4062 17.7273 47.9573C17.7273 49.5084 16.4045 50.7658 14.7727 50.7658Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M26.5909 50.7658C24.9592 50.7658 23.6364 49.5084 23.6364 47.9573C23.6364 46.4062 24.9592 45.1487 26.5909 45.1487C28.2227 45.1487 29.5455 46.4062 29.5455 47.9573C29.5455 49.5084 28.2227 50.7658 26.5909 50.7658Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M50.2273 50.7658C48.5955 50.7658 47.2727 49.5084 47.2727 47.9573C47.2727 46.4062 48.5955 45.1487 50.2273 45.1487C51.859 45.1487 53.1818 46.4062 53.1818 47.9573C53.1818 49.5084 51.859 50.7658 50.2273 50.7658Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M62.0455 50.7658C60.4137 50.7658 59.0909 49.5084 59.0909 47.9573C59.0909 46.4062 60.4137 45.1487 62.0455 45.1487C63.6772 45.1487 65 46.4062 65 47.9573C65 49.5084 63.6772 50.7658 62.0455 50.7658Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M2.95455 39.5316C1.3228 39.5316 7.04798e-08 38.2742 0 36.7231C-7.04798e-08 35.172 1.32279 33.9145 2.95455 33.9145C4.5863 33.9145 5.90909 35.172 5.90909 36.7231C5.90909 38.2742 4.5863 39.5316 2.95455 39.5316Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M26.5909 39.5316C24.9592 39.5316 23.6364 38.2742 23.6364 36.7231C23.6364 35.172 24.9592 33.9145 26.5909 33.9145C28.2227 33.9145 29.5455 35.172 29.5455 36.7231C29.5455 38.2742 28.2227 39.5316 26.5909 39.5316Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M38.4091 39.5316C36.7773 39.5316 35.4545 38.2742 35.4545 36.7231C35.4545 35.172 36.7773 33.9145 38.4091 33.9145C40.0408 33.9145 41.3636 35.172 41.3636 36.7231C41.3636 38.2742 40.0408 39.5316 38.4091 39.5316Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M50.2273 39.5316C48.5955 39.5316 47.2727 38.2742 47.2727 36.7231C47.2727 35.172 48.5955 33.9145 50.2273 33.9145C51.859 33.9145 53.1818 35.172 53.1818 36.7231C53.1818 38.2742 51.859 39.5316 50.2273 39.5316Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M62.0455 39.5316C60.4137 39.5316 59.0909 38.2742 59.0909 36.7231C59.0909 35.172 60.4137 33.9145 62.0455 33.9145C63.6772 33.9145 65 35.172 65 36.7231C65 38.2742 63.6772 39.5316 62.0455 39.5316Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M38.4091 50.7658C36.7773 50.7658 35.4545 49.5084 35.4545 47.9573C35.4545 46.4062 36.7773 45.1487 38.4091 45.1487C40.0408 45.1487 41.3636 46.4062 41.3636 47.9573C41.3636 49.5084 40.0408 50.7658 38.4091 50.7658Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M14.7727 39.5316C13.141 39.5316 11.8182 38.2742 11.8182 36.7231C11.8182 35.172 13.141 33.9145 14.7727 33.9145C16.4045 33.9145 17.7273 35.172 17.7273 36.7231C17.7273 38.2742 16.4045 39.5316 14.7727 39.5316Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M26.5909 62C24.9592 62 23.6364 60.7426 23.6364 59.1915C23.6364 57.6403 24.9592 56.3829 26.5909 56.3829C28.2227 56.3829 29.5455 57.6403 29.5455 59.1915C29.5455 60.7426 28.2227 62 26.5909 62Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M62.0455 62C60.4137 62 59.0909 60.7426 59.0909 59.1915C59.0909 57.6403 60.4137 56.3829 62.0455 56.3829C63.6772 56.3829 65 57.6403 65 59.1915C65 60.7426 63.6772 62 62.0455 62Z" fill="#1D2977" fill-opacity="0.12" />
                                    <path d="M2.95455 62C1.3228 62 1.0914e-06 60.7426 1.02092e-06 59.1915C9.5044e-07 57.6403 1.3228 56.3829 2.95455 56.3829C4.5863 56.3829 5.90909 57.6403 5.90909 59.1915C5.90909 60.7426 4.5863 62 2.95455 62Z" fill="#1D2977" fill-opacity="0.12" />
                                </svg>
                            </div>
                            <p><?php echo esc_html($first_brand->description ?: 'مبلمان‌هایی با طراحی خاص، متریال باکیفیت و رنگ‌بندی منحصربه‌فرد که جلوه‌ای متفاوت و لوکس به فضای خانه شما می‌بخشند.'); ?></p>
                            <a href="<?php echo esc_url(get_term_link($first_brand)); ?>">
                                <span><?php echo __("مشاهده محصولات", "lamkadeh") ?></span>
                                <svg width="34" height="34" viewBox="0 0 34 34" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M20.0747 11.4905C19.9404 11.6248 19.7565 11.7097 19.5444 11.7097L11.7096 11.7097L11.7096 19.5444C11.7096 19.9546 11.3702 20.294 10.9601 20.294C10.55 20.294 10.2106 19.9546 10.2106 19.5444L10.2106 10.9602C10.2106 10.55 10.55 10.2106 10.9601 10.2106L19.5444 10.2106C19.9545 10.2106 20.2939 10.55 20.2939 10.9602C20.301 11.1652 20.2091 11.3561 20.0747 11.4905Z" fill="white" />
                                    <path d="M23.5111 23.5114C23.2212 23.8013 22.7404 23.8013 22.4505 23.5114L10.5499 11.6108C10.2599 11.3208 10.2599 10.84 10.5499 10.5501C10.8398 10.2602 11.3206 10.2602 11.6105 10.5501L23.5111 22.4507C23.801 22.7406 23.801 23.2214 23.5111 23.5114Z" fill="white" />
                                </svg>
                            </a>
                            <svg class="rectangle-svg" width="65" height="41" viewBox="0 0 65 41" fill="none" xmlns="http://www.w3.org/2000/svg">
                            </svg>
                        </div>

                        <div class="product-brands-list-left container">
                            <div class="product-brands-list-left-header">
                                <button class="brands-scroll-btn left" aria-label="scroll left">
                                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M13.6801 18.0701C13.6801 17.8801 13.7501 17.6901 13.9001 17.5401L19.4401 12.0001L13.9001 6.46012C13.6101 6.17012 13.6101 5.69012 13.9001 5.40012C14.1901 5.11012 14.6701 5.11012 14.9601 5.40012L21.0301 11.4701C21.3201 11.7601 21.3201 12.2401 21.0301 12.5301L14.9601 18.6001C14.6701 18.8901 14.1901 18.8901 13.9001 18.6001C13.7501 18.4601 13.6801 18.2601 13.6801 18.0701Z" fill="#1D2977" />
                                        <path d="M2.75008 12C2.75008 11.59 3.09008 11.25 3.50008 11.25L20.3301 11.25C20.7401 11.25 21.0801 11.59 21.0801 12C21.0801 12.41 20.7401 12.75 20.3301 12.75L3.50008 12.75C3.09008 12.75 2.75008 12.41 2.75008 12Z" fill="#1D2977" />
                                    </svg>
                                </button>
                                <div class="product-brands-list-left-header-item-wrapper">
                                    <?php foreach ($brands as $index => $brand): ?>
                                        <div class="product-brands-list-left-header-item <?php echo $index === 0 ? 'active' : ''; ?>"
                                            data-brand-id="<?php echo $brand->term_id; ?>"
                                            data-brand-name="<?php echo esc_attr($brand->name); ?>"
                                            data-brand-description="<?php echo esc_attr($brand->description ?: 'مبلمان‌هایی با طراحی خاص، متریال باکیفیت و رنگ‌بندی منحصربه‌فرد که جلوه‌ای متفاوت و لوکس به فضای خانه شما می‌بخشند.'); ?>"
                                            data-brand-count="<?php echo $brand->count; ?>"
                                            data-brand-link="<?php echo esc_url(get_term_link($brand)); ?>"
                                            data-brand-image="<?php
                                                                $brand_image = get_term_meta($brand->term_id, 'thumbnail_id', true);
                                                                $image_url = $brand_image ? wp_get_attachment_url($brand_image) : 'https://pngdl.ir/wp-content/uploads/2024/07/258851.jpg';
                                                                echo esc_url($image_url);
                                                                ?>">
                                            <?php
                                            $brand_image = get_term_meta($brand->term_id, 'thumbnail_id', true);
                                            $image_url = $brand_image ? wp_get_attachment_url($brand_image) : 'https://pngdl.ir/wp-content/uploads/2024/07/258851.jpg';
                                            ?>
                                            <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($brand->name); ?>">
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                                <button class="brands-scroll-btn right" aria-label="scroll right">
                                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M10.3199 18.0701C10.3199 17.8801 10.2499 17.6901 10.0999 17.5401L4.55994 12.0001L10.0999 6.46012C10.3899 6.17012 10.3899 5.69012 10.0999 5.40012C9.80994 5.11012 9.32994 5.11012 9.03994 5.40012L2.96994 11.4701C2.67994 11.7601 2.67994 12.2401 2.96994 12.5301L9.03994 18.6001C9.32994 18.8901 9.80994 18.8901 10.0999 18.6001C10.2499 18.4601 10.3199 18.2601 10.3199 18.0701Z" fill="#1D2977" />
                                        <path d="M21.2499 12C21.2499 11.59 20.9099 11.25 20.4999 11.25L3.66992 11.25C3.25992 11.25 2.91992 11.59 2.91992 12C2.91992 12.41 3.25992 12.75 3.66992 12.75L20.4999 12.75C20.9099 12.75 21.2499 12.41 21.2499 12Z" fill="#1D2977" />
                                    </svg>
                                </button>
                            </div>

                            <div class="product-brands-list-left-bottom">
                                <?php
                                $first_brand_products = get_brand_products($first_brand->term_id);
                                display_products($first_brand_products, $first_brand->term_id);
                                ?>
                            </div>
                        </div>

                        <script>
                            document.addEventListener('DOMContentLoaded', function() {
                                const brandItems = document.querySelectorAll('.product-brands-list-left-header-item');
                                const productsContainer = document.querySelector('.product-brands-list-left-bottom');

                                brandItems.forEach(item => {
                                    item.addEventListener('click', function() {
                                        brandItems.forEach(brand => brand.classList.remove('active'));

                                        this.classList.add('active');

                                        const brandId = this.dataset.brandId;

                                        updateRightSection(this);

                                        updateProducts(brandId);
                                    });
                                });

                                function updateRightSection(selectedBrand) {
                                    const rightSection = document.querySelector('.product-brands-list-right');
                                    const brandImage = rightSection.querySelector('img');
                                    const productCount = rightSection.querySelector('.product-brands-list-right-header-button strong');
                                    const description = rightSection.querySelector('p');
                                    const link = rightSection.querySelector('a');

                                    brandImage.src = selectedBrand.dataset.brandImage;
                                    brandImage.alt = selectedBrand.dataset.brandName;
                                    productCount.textContent = selectedBrand.dataset.brandCount + '+';
                                    description.textContent = selectedBrand.dataset.brandDescription;
                                    link.href = selectedBrand.dataset.brandLink;
                                }

                                function updateProducts(brandId) {
                                    // نمایش loading
                                    productsContainer.innerHTML = '<div class="loading">در حال بارگذاری محصولات...</div>';

                                    // ارسال درخواست AJAX
                                    fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                                            method: 'POST',
                                            headers: {
                                                'Content-Type': 'application/x-www-form-urlencoded',
                                            },
                                            body: 'action=get_brand_products&brand_id=' + brandId
                                        })
                                        .then(response => {
                                            if (!response.ok) {
                                                throw new Error('Network response was not ok');
                                            }
                                            return response.text();
                                        })
                                        .then(data => {
                                            // بررسی اینکه پاسخ معتبر است
                                            try {
                                                const jsonData = JSON.parse(data);
                                                if (jsonData.success) {
                                                    productsContainer.innerHTML = jsonData.data;
                                                } else {
                                                    productsContainer.innerHTML = '<div class="error">خطا در بارگذاری محصولات: ' + (jsonData.data || 'خطای ناشناخته') + '</div>';
                                                }
                                            } catch (e) {
                                                console.error('JSON Parse Error:', e);
                                                console.log('Raw response:', data);
                                                productsContainer.innerHTML = '<div class="error">پاسخ سرور نامعتبر است</div>';
                                            }
                                        })
                                        .catch(error => {
                                            console.error('Error:', error);
                                            productsContainer.innerHTML = '<div class="error">خطا در اتصال به سرور</div>';
                                        });
                                }
                            });
                        </script>

                    <?php } else { ?>
                        <p>هیچ برندی یافت نشد.</p>
                    <?php } ?>
                </div>
            </section>
        </section>
<?php
    }
}
